#!/usr/bin/env bash

set -eo pipefail

export AWS_STS_REGIONAL_ENDPOINTS=regional
export AWS_SDK_LOAD_CONFIG=true

usage() {
    cat <<EOF
    "Usage: $0 -r <region> -e <environment> <command> [args ...]"

    <region>
        The name of the region to build in. E.g. us-west-2

    <environment>
        The environment to use. E.g. production, dev, etc...

    <command> [args ...]
    In general, these are command and args that are passed to terraform binary as
    arguments

    E.g.
      $0 -r <region> -e <environment> plan

      gets passed to terraform as

      terraform plan
EOF
    exit 1
}

requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && usage )
}

tf_init() {
    shift

    rm -f .terraform/modules/modules.json
    terraform init "$@" -reconfigure -backend-config="modules/${environment}/${region}/backend.tfvars"
}

while getopts 'hr:e:' option; do
    case "$option" in
        r)
            region="$OPTARG"
            ;;
        e)
            environment="$OPTARG"
            ;;
        h)
            usage
            ;;
        *)
            ;;
    esac
done

## Make sure everything is set
[[ -z "$environment" || -z "$region" ]] && usage

requirements

shift "$((OPTIND-1))"

case "$1" in
    init)
        tf_init "$@"
        ;;
    plan|apply|refresh)
        tf_init "$@"
        exec terraform "$@" \
            -var="region=${region}" \
            -var="env=${environment}" \
            -var-file="modules/${environment}/${region}/variables.tfvars"
        ;;
    *)
        exec terraform "$@"
        ;;
esac
