# Creates an ECS cluster, an ECS service for Nginx,
# and an nginx and aptmirror task definition

resource "aws_ecs_cluster" "mrpr_cluster" {
  name = "mrpr_cluster"

  setting {
    name  = "containerInsights"
    value = "enabled"
  }

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-ecs-cluster"
    )
  )
}

resource "aws_ecs_service" "mrpr_nginx_service" {
  name             = "nginx-package-serving"
  cluster          = aws_ecs_cluster.mrpr_cluster.id
  task_definition  = aws_ecs_task_definition.mrpr_nginx_def.arn
  desired_count    = var.nginx_desired_task_count
  launch_type      = "FARGATE"
  platform_version = "1.4.0" #This MUST be specified 1.4.0 or greater to work with fargate. "LATEST" actually defaults to 1.3.0

  network_configuration {
    security_groups  = [aws_security_group.ecs_sg.id]
    subnets          = module.mrpr_vpc.private_subnet_ids
    assign_public_ip = false
  }

  load_balancer {
    target_group_arn = aws_lb_target_group.mrpr_alb_tg.arn
    container_name   = "nginx_container_def"
    container_port   = 80
  }

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-nginx-service"
    )
  )
}

resource "aws_ecs_task_definition" "mrpr_nginx_def" {
  family                   = "mrpr_nginx_def"
  requires_compatibilities = ["FARGATE"]
  cpu                      = var.nginx_cpu_allocation
  memory                   = var.nginx_memory_allocation
  network_mode             = "awsvpc"
  execution_role_arn       = aws_iam_role.execution_role.arn
  task_role_arn            = aws_iam_role.execution_role.arn

  container_definitions = <<DEFINITION
    [{
      "name": "nginx_container_def",
      "image": "${aws_ecr_repository.nginx_repo.repository_url}:${var.nginx_image_version}",
      "essential": true,
      "linuxParameters":
        {
          "initProcessEnabled": true
        },
      "portMappings": [
        {
          "containerPort": 80
        }
      ],
      "mountPoints": [
          {
            "containerPath": "/mnt/aptmirror",
            "sourceVolume": "nginx-container"
          }
      ],
      "logConfiguration":
          {
            "logDriver": "awslogs",
            "options":
              {
                "awslogs-group": "${aws_cloudwatch_log_group.nginx_log_group.name}",
                "awslogs-region": "${var.region}",
                "awslogs-stream-prefix": "nginx-task"
              }
          }
    }]
    DEFINITION

  volume {
    name = "nginx-container"
    efs_volume_configuration {
      file_system_id = aws_efs_file_system.mrpr_efs.id
    }
  }

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-nginx-definition"
    )
  )
}

resource "aws_ecs_task_definition" "mrpr_aptmirror_def" {
  family                   = "mrpr_aptmirror_def"
  requires_compatibilities = ["FARGATE"]
  cpu                      = "256"
  memory                   = "2048"
  network_mode             = "awsvpc"
  execution_role_arn       = aws_iam_role.execution_role.arn
  task_role_arn            = aws_iam_role.execution_role.arn

  container_definitions = <<DEFINITION
      [{
        "name": "aptmirror_container_def",
        "image": "${aws_ecr_repository.aptmirror_repo.repository_url}:${var.aptmirror_image_version}",
        "essential": true,
        "portMappings": [
          {
            "containerPort": 80
          }
        ],
        "mountPoints": [
            {
              "containerPath": "/mnt/aptmirror",
              "sourceVolume": "aptmirror-container"
            }
        ],
        "secrets": [
          {
            "name": "UA_KEY_V1",
            "valueFrom": "${aws_secretsmanager_secret.ubuntu_advantage_key.arn}"
          }
        ],
        "logConfiguration":
          {
            "logDriver": "awslogs",
            "options":
              {
                "awslogs-group": "${aws_cloudwatch_log_group.aptmirror_log_group.name}",
                "awslogs-region": "${var.region}",
                "awslogs-stream-prefix": "aptmirror-task"
              }
          }
      }]
      DEFINITION

  volume {
    name = "aptmirror-container"
    efs_volume_configuration {
      file_system_id = aws_efs_file_system.mrpr_efs.id
    }
  }

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-aptmirror-definition"
    )
  )
}
