# Creates a Task Execution role and the policy for it

resource "aws_iam_role" "execution_role" {
  name               = "${local.service_name}-execution-role"
  assume_role_policy = data.aws_iam_policy_document.assume_role_policy.json

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-execution-role"
    )
  )
}

data "aws_iam_policy_document" "assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

resource "aws_iam_policy" "policy" {
  name   = "${local.service_name}-execution-role-policy"
  path   = "/"
  policy = data.aws_iam_policy_document.execution_role_doc.json
}

data "aws_iam_policy_document" "execution_role_doc" {
  statement {
    effect = "Allow"
    actions = [
      "ecr:GetAuthorizationToken",
      "ecr:BatchCheckLayerAvailability",
      "ecr:GetDownloadUrlForLayer",
      "ecr:BatchGetImage",
      "ecs:CreateCluster",
      "ecs:CreateService",
      "ecs:DeregisterContainerInstance",
      "ecs:DiscoverPollEndpoint",
      "ecs:Poll",
      "ecs:RegisterContainerInstance",
      "ecs:RunTask",
      "ecs:StartTelemetrySession",
      "ecs:StartTask",
      "ecs:StopTask",
      "ecs:UpdateContainerInstancesState",
      "ecs:Submit*",
      "iam:PassRole",
      "logs:CreateLogGroup",
      "logs:CreateLogStream",
      "logs:PutLogEvents",
      "secretsmanager:GetSecretValue",
    ]

    resources = [
      "*",
    ]
  }

  # This allows SSM access for ECS Exec
  statement {
    effect = "Allow"
    actions = [
      "ssmmessages:CreateControlChannel",
      "ssmmessages:CreateDataChannel",
      "ssmmessages:OpenControlChannel",
      "ssmmessages:OpenDataChannel",
    ]

    resources = [
      "*"
    ]
  }
}

resource "aws_iam_policy_attachment" "execution_role_policy_attachment" {
  name       = "execution_role_attachment"
  roles      = [aws_iam_role.execution_role.name]
  policy_arn = aws_iam_policy.policy.arn
}




# Creates a role and policy to enable pull/push to ECR
resource "aws_iam_role" "package_repo_pull_role" {
  name               = "package-repo-pull-role"
  assume_role_policy = module.video_org_principal_destination.policy.json
}

data "aws_iam_policy_document" "allow_ecr_pull" {
  statement {
    effect = "Allow"
    actions = [
      "ecr:GetDownloadUrlForLayer",
      "ecr:BatchGetImage",
      "ecr:BatchCheckLayerAvailability"
    ]
    resources = [
      aws_ecr_repository.aptmirror_repo.arn,
      aws_ecr_repository.nginx_repo.arn,
    ]
  }
}

resource "aws_iam_policy" "role_policy_allow_ecr_pull" {
  name   = "package-repo-allow-ecr-pull"
  policy = data.aws_iam_policy_document.allow_ecr_pull.json
}

resource "aws_iam_role_policy_attachment" "package_repo_pull_role_attachement" {
  role       = aws_iam_role.package_repo_pull_role.id
  policy_arn = aws_iam_policy.role_policy_allow_ecr_pull.arn
}
