# Creates the ALB that fronts the service, and
# the NLB that fronts the ALB for the purposes of
# DX connectivity via VPCe

# Declared once to listen on port 80...
module "mrpr_vpc_endpoint_service" {
  source              = "git@git.xarth.tv:video-coreservices/tf-mod-vpce-service.git?ref=v2.0.0"
  service_id          = "mrpr"
  region              = var.region
  nlb_arn             = aws_lb.mrpr_vpce_nlb.arn
  nlb_tg_arn          = aws_lb_target_group.mrpr_vpce_nlb_tg_80.arn
  alb_arn             = aws_lb.package_repo_alb.arn
  alb_listener_port   = 80
  allowed_principals  = var.vpce_allowed_principals
  enable_vpce_service = true
}

# ...and declared a second time for port 443
module "mrpr_vpc_endpoint_service_443" {
  source              = "git@git.xarth.tv:video-coreservices/tf-mod-vpce-service.git?ref=v2.0.0"
  service_id          = "mrpr-443"
  region              = var.region
  nlb_arn             = aws_lb.mrpr_vpce_nlb.arn
  nlb_tg_arn          = aws_lb_target_group.mrpr_vpce_nlb_tg_443.arn
  alb_arn             = aws_lb.package_repo_alb.arn
  alb_listener_port   = 443
  allowed_principals  = var.vpce_allowed_principals
  enable_vpce_service = false
}

# NLB for the VPCe
resource "aws_lb" "mrpr_vpce_nlb" {
  name               = "${local.service_name}-${var.region}-nlb"
  internal           = true
  load_balancer_type = "network"
  subnets            = module.mrpr_vpc.private_subnet_ids

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-nlb"
    )
  )
}

# 443 listener for the NLB.
resource "aws_lb_listener" "mrpr_vpce_nlb_https_listener" {
  load_balancer_arn = aws_lb.mrpr_vpce_nlb.id
  port              = 443
  protocol          = "TCP"

  default_action {
    target_group_arn = aws_lb_target_group.mrpr_vpce_nlb_tg_443.arn
    type             = "forward"
  }
}

# 80 listener for the NLB
resource "aws_lb_listener" "mrpr_vpce_nlb_http_listener" {
  load_balancer_arn = aws_lb.mrpr_vpce_nlb.id
  port              = 80
  protocol          = "TCP"

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.mrpr_vpce_nlb_tg_80.arn
  }
}

# Target group that will get updated by the lambda created by the module.
resource "aws_lb_target_group" "mrpr_vpce_nlb_tg_80" {
  name        = "package-repo-vpce-nlb-tg-80"
  port        = 80
  protocol    = "TCP"
  vpc_id      = module.mrpr_vpc.vpc_id
  target_type = "ip"

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-vpce-nlb-tg"
    )
  )
}

# Target group that will get updated by the lambda created by the module.
resource "aws_lb_target_group" "mrpr_vpce_nlb_tg_443" {
  name        = "package-repo-vpce-nlb-tg-443"
  port        = 443
  protocol    = "TCP"
  vpc_id      = module.mrpr_vpc.vpc_id
  target_type = "ip"

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-vpce-nlb-tg"
    )
  )
}

# We want to set up an ALB that exposes port 80
resource "aws_lb" "package_repo_alb" {
  name                       = "${local.service_name}-${var.region}-alb"
  internal                   = true
  load_balancer_type         = "application"
  security_groups            = [aws_security_group.target_sg.id]
  subnets                    = module.mrpr_vpc.private_subnet_ids
  enable_deletion_protection = false

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-alb"
    )
  )
}

# Target group to forward listener traffic to the listening port nginx is configured to listen on
resource "aws_lb_target_group" "mrpr_alb_tg" {
  name        = "package-repo-alb-tg"
  port        = 80
  protocol    = "HTTP"
  vpc_id      = module.mrpr_vpc.vpc_id
  target_type = "ip"

  health_check {
    protocol            = "HTTP"
    path                = "/status"
    interval            = 10
    healthy_threshold   = 3
    unhealthy_threshold = 3
  }

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-alb"
    )
  )
}

resource "aws_lb_listener" "mrpr_http_listener" {
  load_balancer_arn = aws_lb.package_repo_alb.arn
  port              = 80
  protocol          = "HTTP"

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.mrpr_alb_tg.arn
  }
}

resource "aws_lb_listener" "mrpr_https_listener" {
  load_balancer_arn = aws_lb.package_repo_alb.arn
  port              = 443
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2016-08"
  certificate_arn   = data.terraform_remote_state.acm_backend.outputs.mrpr_cert_arn

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.mrpr_alb_tg.arn
  }
}

data "terraform_remote_state" "acm_backend" {
  backend = "s3"

  config = {
    bucket = var.backend_bucket
    key    = "terraform_bootstrap_acm.tfstate"
    region = var.region
  }
}
