# All monitoring for various resources

module "sns_topic_mod" {
  source       = "git::git+ssh://git@git.xarth.tv/video-coreservices/tf-mod-sns-subscription?ref=v1.0.0"
  service_name = "package-repo-${var.region}"
}


# ALB - One or fewer healthy hosts
resource "aws_cloudwatch_metric_alarm" "alb_unhealthy" {
  alarm_name          = "mrpr-alb-${var.region}-failing"
  metric_name         = "HealthyHostCount"
  namespace           = "AWS/ApplicationELB"
  comparison_operator = "LessThanOrEqualToThreshold"
  threshold           = "1"
  evaluation_periods  = "3"
  period              = "300"
  statistic           = "Average"
  alarm_description   = "There are one or fewer healthy Nginx servers behind the ALB in ${var.region}. https://wiki.xarth.tv/display/VID/Package+Repo"
  alarm_actions       = [module.sns_topic_mod.business_hours_topic_arn]
  ok_actions          = [module.sns_topic_mod.business_hours_topic_arn]

  dimensions = {
    TargetGroup  = aws_lb_target_group.mrpr_alb_tg.arn_suffix
    LoadBalancer = aws_lb.package_repo_alb.arn_suffix
  }
}

# NLB - One or fewer healthy hosts
resource "aws_cloudwatch_metric_alarm" "nlb_unhealthy" {
  alarm_name          = "mrpr-nlb-${var.region}-failing"
  metric_name         = "HealthyHostCount"
  namespace           = "AWS/NetworkELB"
  comparison_operator = "LessThanOrEqualToThreshold"
  threshold           = "1"
  evaluation_periods  = "3"
  period              = "300"
  statistic           = "Average"
  alarm_description   = "There are one or fewer healthy Nginx servers behind the consul NLB in ${var.region}. https://wiki.xarth.tv/display/VID/Package+Repo"
  alarm_actions       = [module.sns_topic_mod.business_hours_topic_arn]
  ok_actions          = [module.sns_topic_mod.business_hours_topic_arn]

  dimensions = {
    TargetGroup  = aws_lb_target_group.mrpr_vpce_nlb_tg_80.arn_suffix
    LoadBalancer = aws_lb.mrpr_vpce_nlb.arn_suffix
  }
}

# Running Nginx task count too low
resource "aws_cloudwatch_metric_alarm" "nginx_task_low" {
  alarm_name          = "nginx-${var.region}-task-count-too-low"
  metric_name         = "RunningTaskCount"
  namespace           = "ECS/ContainerInsights"
  comparison_operator = "LessThanOrEqualToThreshold"
  threshold           = "1"
  evaluation_periods  = "3"
  period              = "300"
  statistic           = "Average"
  alarm_description   = "There are 1 or fewer Nginx tasks running in ${var.region}. https://wiki.xarth.tv/display/VID/Package+Repo"
  alarm_actions       = [module.sns_topic_mod.business_hours_topic_arn]
  ok_actions          = [module.sns_topic_mod.business_hours_topic_arn]

  dimensions = {
    ClusterName = aws_ecs_cluster.mrpr_cluster.name
    ServiceName = aws_ecs_service.mrpr_nginx_service.name
  }
}

resource "aws_cloudwatch_log_group" "aptmirror_log_group" {
  name = "aptmirror-Log-Group"

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-aptmirror-log-group"
    )
  )
}

resource "aws_cloudwatch_log_group" "nginx_log_group" {
  name = "nginx-Log-Group"

  tags = merge(
    local.common_tags,
    map(
      "Name", "${local.service_name}-nginx-log-group"
    )
  )
}

locals {
  metric_transformation_name      = "AptmirrorErrorCount"
  metric_transformation_namespace = "Aptmirror"
}

resource "aws_cloudwatch_log_metric_filter" "aptmirror_error_filter" {
  name           = "aptmirror-error-filter-${var.region}"
  pattern        = "?Err ?Failed ?401 Unauthorized"
  log_group_name = aws_cloudwatch_log_group.aptmirror_log_group.name

  metric_transformation {
    name          = local.metric_transformation_name
    namespace     = local.metric_transformation_namespace
    value         = "1"
    default_value = "0"
  }
}

resource "aws_cloudwatch_metric_alarm" "aptmirror_error_alarm" {
  alarm_name          = "${var.region}-aptmirror-error"
  metric_name         = local.metric_transformation_name
  namespace           = local.metric_transformation_namespace
  comparison_operator = "GreaterThanOrEqualToThreshold"
  threshold           = "1"
  evaluation_periods  = "1"
  period              = "86400"
  statistic           = "Average"
  alarm_description   = "Apt-mirror may have failed in ${var.region}. https://wiki.xarth.tv/display/VID/Package+Repo"
  alarm_actions       = [module.sns_topic_mod.business_hours_topic_arn]
  ok_actions          = [module.sns_topic_mod.business_hours_topic_arn]
}
