data "template_cloudinit_config" "userdata" {
  gzip          = false
  base64_encode = false

  part {
    filename     = "init.cfg"
    content_type = "text/cloud-config"

    content = templatefile("${path.module}/templates/cloudinit.yaml.tmpl", {
      service    = "video-puppet-ca"
      dns_region = var.region
      dns_zone   = replace(module.variables.svc_puppet_r53_zone_name, "${var.region}.", "")
    })
  }

  part {
    content_type = "text/x-shellscript"

    content = templatefile("${path.module}/templates/startup.sh.tmpl", {
      environment = local.environment
      owner       = "vidcs"
      service     = "video-puppet-ca"
      dns_region  = var.region
      dns_zone    = replace(module.variables.svc_puppet_r53_zone_name, "${var.region}.", "")

      # Because we are unable to use AWS magic DNS to auto resolve
      nfs_ips = format(" %s,%s %s,%s %s,%s ",
        aws_efs_mount_target.efs_az_a.subnet_id, aws_efs_mount_target.efs_az_a.ip_address,
        aws_efs_mount_target.efs_az_b.subnet_id, aws_efs_mount_target.efs_az_b.ip_address,
        aws_efs_mount_target.efs_az_c.subnet_id, aws_efs_mount_target.efs_az_c.ip_address,
      )
    })
  }
}

resource "aws_iam_instance_profile" "instance_profile" {
  name = "video-puppet-ca-${var.env}"
  role = aws_iam_role.role.name
}

data "aws_iam_policy_document" "instance_assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "role" {
  name               = "video-puppet-ca-${var.env}"
  assume_role_policy = data.aws_iam_policy_document.instance_assume_role_policy.json
}

data "aws_ami" "base_ami" {
  most_recent = true
  owners      = ["043714768218"]

  filter {
    name   = "name"
    values = ["devtools-baseami-xenial-consulfree-*"]
  }
}

resource "aws_launch_configuration" "launch_conf" {
  name_prefix          = "video-puppet-ca_launch_config"
  image_id             = data.aws_ami.base_ami.id
  instance_type        = var.instance_type
  iam_instance_profile = aws_iam_instance_profile.instance_profile.name
  security_groups = [
    module.variables.sg_twitch_subnet_id,
    aws_security_group.puppet_ca_servers_sg.id,
  ]
  user_data = data.template_cloudinit_config.userdata.rendered

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "asg" {
  name                 = "video-puppet-ca_asg"
  vpc_zone_identifier  = module.variables.private_subnet_ids
  max_size             = 1
  min_size             = 1
  health_check_type    = "ELB"
  launch_configuration = aws_launch_configuration.launch_conf.name
  target_group_arns = [
    aws_lb_target_group.puppet_tg.arn,
    aws_lb_target_group.ssh_tg.arn,
  ]

  depends_on = [
    aws_efs_mount_target.efs_az_a,
    aws_efs_mount_target.efs_az_b,
    aws_efs_mount_target.efs_az_c,
  ]

  dynamic "tag" {
    for_each = merge({
      Name                   = "video-puppet-ca",
      "asg:hostname_pattern" = "video-puppet-ca-#instanceid.${module.variables.svc_puppet_r53_zone_name}@${module.variables.svc_puppet_r53_zone_id}"
    }, local.common_tags)

    content {
      key                 = tag.key
      value               = tag.value
      propagate_at_launch = true
    }
  }
}

# Handle ASG DNS
resource "aws_autoscaling_lifecycle_hook" "launching" {
  name                    = "lifecycle-launching"
  autoscaling_group_name  = aws_autoscaling_group.asg.name
  default_result          = "ABANDON"
  heartbeat_timeout       = 60
  lifecycle_transition    = "autoscaling:EC2_INSTANCE_LAUNCHING"
  notification_target_arn = module.autoscale_dns.autoscale_handling_sns_topic_arn
  role_arn                = module.autoscale_dns.agent_lifecycle_iam_role_arn
}

resource "aws_autoscaling_lifecycle_hook" "terminating" {
  name                    = "lifecycle-terminating"
  autoscaling_group_name  = aws_autoscaling_group.asg.name
  default_result          = "CONTINUE"
  heartbeat_timeout       = 60
  lifecycle_transition    = "autoscaling:EC2_INSTANCE_TERMINATING"
  notification_target_arn = module.autoscale_dns.autoscale_handling_sns_topic_arn
  role_arn                = module.autoscale_dns.agent_lifecycle_iam_role_arn
}

module "autoscale_dns" {
  source = "./vendor/modules/terraform-aws-asg-dns-handler"

  autoscale_handler_unique_identifier = "video-puppet-ca"
  vpc_name                            = module.variables.vpc_id
  autoscale_route53zone_id            = module.variables.svc_puppet_r53_zone_id
}
