package config

import (
	"context"
	"os"

	"code.justin.tv/video-tools/mck"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/pkg/errors"
)

// Config holds the basic configuration for the stats uploader.
type Config struct {
	Mck                 mck.MCK
	Hostname            string
	PuppetDisabledFile  string
	PuppetStateFile     string
	PuppetStateMetrics  []MetricConfig
	PuppetExitCode      int
	AWSRegion           string
	Session             *session.Session
	CloudwatchNamespace string
}

type MetricConfig struct {
	MetricName string
	MetricUnit string
}

// SetupConfig : Setup basic configuration that feeds the metrics pusher
func SetupConfig(code int, region string) (c Config, err error) {
	cfg := defaultConfig()
	// override region if specified
	cfg.PuppetExitCode = code
	if region != "" {
		cfg.AWSRegion = region
	}
	facterMck, err := gatherMCKData()
	if err != nil {
		return cfg, errors.Wrap(err, "failed to gather MCK data")
	}
	cfg.Mck = *facterMck
	hostname, err := os.Hostname()
	if err != nil {
		return cfg, errors.Wrap(err, "failed to read hostname")
	}
	cfg.Hostname = hostname
	// setup AWS fixins'
	sess := session.Must(session.NewSession(&aws.Config{
		Region:              aws.String(cfg.AWSRegion),
		STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
	}))

	cfg.Session = sess
	return cfg, nil
}

// return default config
func defaultConfig() (c Config) {
	return Config{
		Mck:                mck.MCK{},
		PuppetStateFile:    "/var/lib/puppet/state/last_run_summary.yaml",
		PuppetDisabledFile: "/var/lib/puppet/state/agent_disabled.lock",
		PuppetStateMetrics: []MetricConfig{
			{"resources.changed", "Count"},
			{"resources.failed", "Count"},
			{"time.total", "Seconds"},
			{"version.config", "None"},
			{"version.puppet", "None"},
		},
		PuppetExitCode:      -1, // this is interpreted as an Unknown result.
		AWSRegion:           "us-west-2",
		Session:             nil,
		CloudwatchNamespace: "StagedPuppet",
	}
}

// Determine MCK info.
func gatherMCKData() (m *mck.MCK, err error) {
	facter := mck.FacterDeterminer{}
	mck, err := facter.DetermineMCK(context.Background())
	if err != nil {
		return mck, errors.Wrap(err, "failed to retrive mck data")
	}
	return mck, nil
}
