package cwupload

import (
	"os"
	"strings"
	"time"

	"code.justin.tv/video-coreservices/tf-io-puppet-staged-deploy/spmu/cmd/internal/config"
	"code.justin.tv/video-coreservices/tf-io-puppet-staged-deploy/spmu/pkg/puppetstats"
	"code.justin.tv/video-tools/mck"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/cenkalti/backoff"
	"github.com/jinzhu/copier"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

// Send : send puppet metrics to cloudwatch
func Send(cfg config.Config) error {
	// form dimensions and metrics
	dims := dimensionStructure(cfg.Hostname, &cfg.Mck)
	metrics, err := metricPkg(cfg, dims)
	if err != nil {
		log.Errorf("failed to gather all supplementary metrics %i", err)
	}
	// create new CW session
	svc := cloudwatch.New(cfg.Session, aws.NewConfig().WithRegion(cfg.AWSRegion))
	// configure backoff and send metrics
	bo := backoff.NewExponentialBackOff()
	bo.MaxElapsedTime = 1 * time.Minute
	bo.Multiplier = 2
	err = backoff.Retry(func() error {
		req, _ := svc.PutMetricDataRequest(&cloudwatch.PutMetricDataInput{
			MetricData: metrics,
			Namespace:  aws.String(cfg.CloudwatchNamespace),
		})
		return req.Send()
	}, bo)

	if err != nil {
		return errors.Wrap(err, "PutMetricDataRequest to cloudwatch failed.")
	}
	return nil
}

// form and return metrics package to send to cloudwatch
func metricPkg(cfg config.Config, dims [][]*cloudwatch.Dimension) ([]*cloudwatch.MetricDatum, error) {
	// determine which metric name to populate depending on puppet exit code
	exitCodeMetric := &cloudwatch.MetricDatum{
		Unit:  aws.String(cloudwatch.StandardUnitCount),
		Value: aws.Float64(1.0),
	}
	exitCodeMetric.MetricName = aws.String(puppetstats.InterpretExitCode(cfg.PuppetExitCode))
	metricsList := rollupMetrics(exitCodeMetric, dims)
	// gather additional state file metrics
	stateFileMetrics, err := getStateMetrics(cfg.PuppetDisabledFile, cfg.PuppetStateFile, cfg.PuppetStateMetrics, dims)
	if err != nil {
		return metricsList, errors.Wrap(err, "partial error: failed to handle state file, but success/failure still logged")
	}
	return append(stateFileMetrics, metricsList...), nil
}

// gather additional metrics from the state file and machine relevant to puppet applies.
// pushed metrics are dynamically configurable in cfg.PuppetStateMetrics
func getStateMetrics(disabledFile string, stateFile string, metrics []config.MetricConfig, dims [][]*cloudwatch.Dimension) ([]*cloudwatch.MetricDatum, error) {
	agentDisabled := 0.0
	if _, err := os.Stat(disabledFile); err == nil {
		agentDisabled = 1.0
	}
	stateMetrics := rollupMetrics(&cloudwatch.MetricDatum{
		MetricName: aws.String("PuppetAgentDisabled"),
		Unit:       aws.String(cloudwatch.StandardUnitCount),
		Value:      aws.Float64(agentDisabled),
	}, dims)
	// Examine and append state file metrics if agent wasn't disabled.
	if agentDisabled == 0.0 {
		puppetStats, err := puppetstats.ReadConfig(stateFile)
		if err != nil {
			return stateMetrics, errors.Wrap(err, "failed to parse config file")
		}
		cwEnums := cloudwatch.StandardUnit_Values()
		// process all configured metrics
		for _, metricCfg := range metrics {
			keys := strings.Split(metricCfg.MetricName, ".") // index 0 is the outer key and index 1 is the inner key
			metricVal, err := puppetstats.Retrieve(puppetStats, keys[0], keys[1])
			if err != nil {
				log.Errorf("error retrieving configured metric %s from puppet summary", metricCfg.MetricName)
				log.Error(err)
				continue
			}
			if contains(cwEnums, metricCfg.MetricUnit) {
				stateMetrics = append(stateMetrics, rollupMetrics(&cloudwatch.MetricDatum{
					MetricName: aws.String(metricCfg.MetricName),
					Unit:       aws.String(metricCfg.MetricUnit),
					Value:      aws.Float64(metricVal),
				}, dims)...)
			} else {
				log.Errorf("invalid unit for configured metric %s - %s", metricCfg.MetricName, metricCfg.MetricUnit)
			}
		}
	}
	return stateMetrics, nil
}

// return list of metrics from list of dimensions
func rollupMetrics(metric *cloudwatch.MetricDatum, dims [][]*cloudwatch.Dimension) []*cloudwatch.MetricDatum {
	metrics := []*cloudwatch.MetricDatum{}
	for _, dim := range dims {
		cpy := &cloudwatch.MetricDatum{}
		copier.Copy(cpy, metric)
		cpy.Dimensions = dim
		metrics = append(metrics, cpy)
	}
	return metrics
}

// dummy str contains function
func contains(s []string, str string) bool {
	for _, v := range s {
		if v == str {
			return true
		}
	}
	return false
}

// form and return dimension structures, including rollups
func dimensionStructure(hostname string, m *mck.MCK) [][]*cloudwatch.Dimension {
	return [][]*cloudwatch.Dimension{
		{
			{
				Name:  aws.String("MachineClass"),
				Value: aws.String(m.MachineClass),
			},
			{
				Name:  aws.String("Pop"),
				Value: aws.String(m.Pop),
			},
			{
				Name:  aws.String("Environment"),
				Value: aws.String(m.Environment),
			},
			{
				Name:  aws.String("Hostname"),
				Value: aws.String(hostname),
			},
		},
		{
			{
				Name:  aws.String("MachineClass"),
				Value: aws.String(m.MachineClass),
			},
			{
				Name:  aws.String("Pop"),
				Value: aws.String(m.Pop),
			},
			{
				Name:  aws.String("Environment"),
				Value: aws.String(m.Environment),
			},
		},
		{
			{
				Name:  aws.String("MachineClass"),
				Value: aws.String(m.MachineClass),
			},
			{
				Name:  aws.String("Environment"),
				Value: aws.String(m.Environment),
			},
		},
	}
}
