package puppetstats

import (
	"io/ioutil"
	"reflect"
	"strconv"
	"strings"

	"github.com/pkg/errors"
	"gopkg.in/yaml.v2"
)

type PuppetLastRunSummary struct {
	Version struct {
		Config string `yaml:"config"`
		Puppet string `yaml:"puppet"`
	} `yaml:"version"`
	Resources struct {
		Changed         int `yaml:"changed"`
		Failed          int `yaml:"failed"`
		FailedToRestart int `yaml:"failed_to_restart"`
		OutOfSync       int `yaml:"out_of_sync"`
		Restarted       int `yaml:"restarted"`
		Scheduled       int `yaml:"scheduled"`
		Skipped         int `yaml:"skipped"`
		Total           int `yaml:"total"`
	} `yaml:"resources"`
	Time struct {
		Alternatives     float64 `yaml:"alternatives"`
		Anchor           float64 `yaml:"anchor"`
		AptKey           float64 `yaml:"apt_key"`
		Archive          float64 `yaml:"archive"`
		Augeas           float64 `yaml:"augeas"`
		ConcatFile       float64 `yaml:"concat_file"`
		ConcatFragment   float64 `yaml:"concat_fragment"`
		ConfigRetrieval  float64 `yaml:"config_retrieval"`
		Cron             float64 `yaml:"cron"`
		Exec             float64 `yaml:"exec"`
		File             float64 `yaml:"file"`
		Filebucket       float64 `yaml:"filebucket"`
		Firewall         float64 `yaml:"firewall"`
		Group            float64 `yaml:"group"`
		Host             float64 `yaml:"host"`
		Ifstate          float64 `yaml:"ifstate"`
		IniSetting       float64 `yaml:"ini_setting"`
		Notify           float64 `yaml:"notify"`
		Package          float64 `yaml:"package"`
		Refacter         float64 `yaml:"refacter"`
		Resources        float64 `yaml:"resources"`
		Schedule         float64 `yaml:"schedule"`
		Service          float64 `yaml:"service"`
		SSHAuthorizedKey float64 `yaml:"ssh_authorized_key"`
		Sysctl           float64 `yaml:"sysctl"`
		SysctlRuntime    float64 `yaml:"sysctl_runtime"`
		Total            float64 `yaml:"total"`
		User             float64 `yaml:"user"`
		LastRun          int     `yaml:"last_run"`
	} `yaml:"time"`
	Changes struct {
		Total int `yaml:"total"`
	} `yaml:"changes"`
	Events struct {
		Failure int `yaml:"failure"`
		Success int `yaml:"success"`
	} `yaml:"events"`
}

// InterpretExitCode - takes an exit code and returns the metric name
func InterpretExitCode(code int) string {
	switch code {
	case 0, 2:
		return "PuppetExecutionSuccess"
	case 1, 4, 6:
		return "PuppetExecutionFailure"
	default:
		return "PuppetExecutionUnknown"
	}
}

// ReadConfig - read a last_run_summary file and return the its yaml struct
func ReadConfig(filename string) (*PuppetLastRunSummary, error) {
	c := &PuppetLastRunSummary{}
	buf, err := ioutil.ReadFile(filename)
	if err != nil {
		return c, errors.Wrap(err, "failed to read input filename")
	}
	err = yaml.Unmarshal(buf, c)
	if err != nil {
		return c, errors.Wrap(err, "failed to unmarshal yaml")
	}

	return c, nil
}

// Retrieve - retrieve a metric field from the puppet last run summary
// Does not handle values in the Version outer key as values are strings.
func Retrieve(lrs *PuppetLastRunSummary, outer string, inner string) (float64, error) {
	uInner := strings.Title(inner)
	switch strings.Title(outer) {
	case "Version":
		r := reflect.ValueOf(lrs.Version)
		f := reflect.Indirect(r).FieldByName(uInner)
		if !f.IsValid() {
			return 0.0, errors.Errorf("unsupported inner key lookup %s", inner)
		}
		converted, err := versionFloat(inner, f.String())
		if err != nil {
			return 0.0, err
		}
		return converted, nil
	case "Resources":
		r := reflect.ValueOf(lrs.Resources)
		f := reflect.Indirect(r).FieldByName(uInner)
		if !f.IsValid() {
			return 0.0, errors.Errorf("unsupported inner key lookup %s", inner)
		}
		return float64(f.Int()), nil
	case "Time":
		r := reflect.ValueOf(lrs.Time)
		f := reflect.Indirect(r).FieldByName(uInner)
		if !f.IsValid() {
			return 0.0, errors.Errorf("unsupported inner key lookup %s", inner)
		}
		return f.Float(), nil
	case "Changes":
		r := reflect.ValueOf(lrs.Changes)
		f := reflect.Indirect(r).FieldByName(uInner)
		if !f.IsValid() {
			return 0.0, errors.Errorf("unsupported inner key lookup %s", inner)
		}
		return float64(f.Int()), nil
	case "Events":
		r := reflect.ValueOf(lrs.Events)
		f := reflect.Indirect(r).FieldByName(uInner)
		if !f.IsValid() {
			return 0.0, errors.Errorf("unsupported inner key lookup %s", inner)
		}
		return float64(f.Int()), nil
	default:
		return 0.0, errors.Errorf("unsupported outer key lookup %s", outer)
	}
}

func versionFloat(key string, field string) (float64, error) {
	switch strings.Title(key) {
	case "Config":
		f, err := strconv.ParseUint(field, 16, 64)
		if err != nil {
			return 0.0, errors.Errorf("unable to parse 'Config' value %s", field)
		}
		return float64(f), nil
	case "Puppet":
		convert := strings.ReplaceAll(field, ".", "0")
		f, err := strconv.ParseFloat(convert, 64)
		if err != nil {
			return 0.0, errors.Errorf("unable to parse 'Puppet' value %s", field)
		}
		return f, nil
	default:
		return 0.0, errors.Errorf("unsupported string conversion field %s", field)
	}
}
