package mck

import (
	"context"
	"errors"
	"fmt"
	"strings"

	"code.justin.tv/video-tools/mck/cmdline"
)

const procCmdlinePath = "/proc/cmdline"

// ProcCmdlineDeterminer implements mck.Determiner by parsing /proc/cmdline obtain
// MCK facts for the host.
type ProcCmdlineDeterminer struct{}

// ensure that ProcCmmdlineDeterminer is a Determiner
var _ Determiner = (*ProcCmdlineDeterminer)(nil)

// DetermineMCK queries facter to obtain the MCK for the host. The facter
// subprocess is tied to the provided context.
func (f *ProcCmdlineDeterminer) DetermineMCK(ctx context.Context) (*MCK, error) {
	parsed, err := cmdline.ParseFile(procCmdlinePath)
	if err != nil {
		return nil, fmt.Errorf("parsing /proc/cmdline: %s", err)
	}

	out := MCK{
		MachineClass: strings.TrimSpace(parsed["MACHINE_CLASS"]),
		Environment:  strings.TrimSpace(parsed["ENVIRONMENT"]),
		Pop:          strings.TrimSpace(parsed["POP"]),
		VTPToken:     strings.TrimSpace(parsed["VTP_TOKEN"]),
	}

	if out.MachineClass == "" && out.Environment == "" && out.Pop == "" && out.VTPToken == "" {
		return nil, errors.New("unable to obtain any environment information from /proc/cmdline")
	}
	return &out, nil
}
