package cmdline

import (
	"fmt"
	"io/ioutil"
	"os"
	"strings"
)

// ParseFile parses a proc cmdline file into a dictionary
func ParseFile(path string) (map[string]string, error) {
	f, err := os.OpenFile(path, os.O_RDONLY, 0644)
	if err != nil {
		return nil, fmt.Errorf("opening file at path:%s, err:%s", path, err)
	}

	data, err := ioutil.ReadAll(f)
	if err != nil {
		return nil, fmt.Errorf("reading file: %s", err)
	}

	if err := f.Close(); err != nil {
		return nil, fmt.Errorf("closing file: %s", err)
	}

	return ParseString(string(data))
}

// ParseString parses data from a proc cmdline file into a dictionary
func ParseString(data string) (map[string]string, error) {
	keyPairs := strings.Split(data, " ")

	parsed := make(map[string]string)
	for _, kp := range keyPairs {
		if !strings.Contains(kp, "=") {
			parsed[kp] = ""
			continue
		}

		splitKP := strings.Split(kp, "=")
		if len(splitKP) != 2 || splitKP[0] == "" {
			return nil, fmt.Errorf("hit unexpected data: %s", kp)
		}
		parsed[splitKP[0]] = splitKP[1]
	}

	return parsed, nil
}
