package mck

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"os"
	"os/exec"
	"strings"
)

// FacterDeterminer implements mck.Determiner by querying facter to obtain the
// MCK for the host.
type FacterDeterminer struct{}

// ensure that FacterDeterminer is a Determiner
var _ Determiner = (*FacterDeterminer)(nil)

// DetermineMCK queries facter to obtain the MCK for the host. The facter
// subprocess is tied to the provided context.
func (f *FacterDeterminer) DetermineMCK(ctx context.Context) (*MCK, error) {
	return fromFacter(ctx)
}

// fromFacter implements querying facter by starting a subprocess tied to the
// provided context.
func fromFacter(ctx context.Context) (*MCK, error) {
	// XXX: consider checking whether the process user is root/superuser and emitting an error if not
	cmd := exec.CommandContext(ctx, "/usr/bin/sudo", "/usr/bin/facter", "--json", "--puppet", "twitch_role", "twitch_environment", "pop", "twitch_pop")
	cmd.Stderr = os.Stderr
	raw, err := cmd.Output()
	if err != nil {
		return nil, fmt.Errorf("unable to run facter, %s", err)
	}
	return fromFacterOutput(raw)
}

// fromFacterOutput translates raw json data to a mck.MCK. The json data is
// assumed to be output from facter. When facter returns a blank value for
// "pop" the value for "twitch_pop" is used.
func fromFacterOutput(raw []byte) (*MCK, error) {
	var parsed map[string]string
	if err := json.Unmarshal(raw, &parsed); err != nil {
		return nil, fmt.Errorf("unable to parse facter output, %s", err)
	}

	out := MCK{
		MachineClass: strings.TrimSpace(parsed["twitch_role"]),
		Environment:  strings.TrimSpace(parsed["twitch_environment"]),
		Pop:          strings.TrimSpace(parsed["pop"]),
	}
	if out.Pop == "" {
		out.Pop = strings.TrimSpace(parsed["twitch_pop"])
	}

	if out.MachineClass == "" && out.Environment == "" && out.Pop == "" {
		return nil, errors.New("unable to obtain any environment information from facter")
	}
	return &out, nil
}
