// Package mck provides ways to obtain the machine classification key (mck) of
// a host.
package mck

import (
	"context"
	"fmt"
)

// MCK is a collection of host identification information.
type MCK struct {

	// MachineClass is the host's type-role or twitch_role or machine class
	// name. It is empty-string if no value is available.
	MachineClass string

	// Environment is the host's stage or twitch_environment or deploy
	// environment name. It is empty-string if no value is available.
	Environment string

	// Pop is the host's pop or twitch_pop or datacenter name. It is
	// empty-string if no value is available.
	Pop string

	// VTPToken is the host's provisioning token, minted by VideoTrustProvider.
	// It is empty-string if no value is available
	VTPToken string
}

// Determiner is the interface for things that can produce an MCK.
type Determiner interface {
	DetermineMCK(cxt context.Context) (*MCK, error)
}

// String returns the canonical string representation of the MCK. It is the machine class, environment, and pop in
// order, delimited by colons (:).
func (m MCK) String() string {
	return fmt.Sprintf("%s:%s:%s", m.MachineClass, m.Environment, m.Pop)
}
