module "default_behavior" {
  source = "git::ssh://git@git.xarth.tv/video-tools/deployment-terraform.git//modules/common/default_behavior?ref=v1.0.7"
}

module "datacenters" {
  source = "git@git.xarth.tv:video-coreservices/tf-mod-datacenters-metadata.git"
}

locals {
  // Shared metadata for deploy accounts. MODIFY METADATA HERE AND DO NOT HARDCODE BELOW
  // name is used for application name and pipeline name
  deploy_acc_metadata = {
    twitch-vidcs-consul-master-prod = {
      account_id      = "625394079058",
      machine_classes = ["consul-master"],
      name            = "consul-master",
    },
    twitch-vidcs-dns-auth-prod = {
      account_id      = "310631158791",
      machine_classes = ["dns-auth"],
      name            = "dns-auth",
    },
    twitch-vidcs-dns-resolver-prod = {
      account_id      = "469262527252",
      machine_classes = ["dns-resolver"],
      name            = "dns-resolver",
    },
    twitch-vidcs-pop-masters-prod = {
      account_id      = "825658541764",
      machine_classes = ["pop-master-a", "pop-master-b"],
      name            = "pop-master",
    },
    twitch-vidcs-squid-proxy-prod = {
      account_id      = "719152506984",
      machine_classes = ["squid-proxy"],
      name            = "squid-proxy",
    },
  }

  // helper locals. These variables are also exposed at the base level of the output map
  // because they are the same value between all deploy accounts.
  control_account_id               = "458835690105"
  artifact_name                    = "puppet-deploy.zip"
  emergency_pipeline_artifact_name = "puppet-emergency-deploy.zip"

  // Adjustable metadata and ordering for edge DC steps.
  // Stages are:
  //  1. first PoP in us-east-2 (smallest region)
  //  2. us-east-2
  //  3. us-west-2
  //  4. ap-southeast-1
  //  5. us-east-1
  //  6. eu-west-1
  // Used by pop-master-[a|b] along with origin_dc_steps
  edge_dc_steps = [
    { name                              = "edge-rollout-canary-${module.datacenters.global_map_by_regions_in_commission["edge"]["us-east-2"][0]}",
      pops                              = [module.datacenters.global_map_by_regions["edge"]["us-east-2"][0]],
      deploy_config_name                = "CodeDeployDefault.OneAtATime",
      max_concurrent_deployment_actions = 1,
    },
    { name                              = "edge-rollout-us-east-2",
      pops                              = slice(module.datacenters.global_map_by_regions["edge"]["us-east-2"], 1, length(module.datacenters.global_map_by_regions_in_commission["edge"]["us-east-2"])),
      deploy_config_name                = "CodeDeployDefault.OneAtATime",
      max_concurrent_deployment_actions = 1,
    },
    { name                              = "edge-rollout-us-west-2",
      pops                              = module.datacenters.global_map_by_regions_in_commission["edge"]["us-west-2"],
      deploy_config_name                = "CodeDeployDefault.OneAtATime",
      max_concurrent_deployment_actions = length(module.datacenters.global_map_by_regions_in_commission["edge"]["us-west-2"]),
    },
    { name                              = "edge-rollout-ap-southeast-1",
      pops                              = module.datacenters.global_map_by_regions_in_commission["edge"]["ap-southeast-1"],
      deploy_config_name                = "CodeDeployDefault.OneAtATime",
      max_concurrent_deployment_actions = length(module.datacenters.global_map_by_regions_in_commission["edge"]["ap-southeast-1"]),
    },
    { name                              = "edge-rollout-us-east-1",
      pops                              = module.datacenters.global_map_by_regions_in_commission["edge"]["us-east-1"],
      deploy_config_name                = "CodeDeployDefault.OneAtATime",
      max_concurrent_deployment_actions = length(module.datacenters.global_map_by_regions_in_commission["edge"]["us-east-1"]),
    },
    { name                              = "edge-rollout-eu-west-1",
      pops                              = module.datacenters.global_map_by_regions_in_commission["edge"]["eu-west-1"],
      deploy_config_name                = "CodeDeployDefault.OneAtATime",
      max_concurrent_deployment_actions = length(module.datacenters.global_map_by_regions_in_commission["edge"]["eu-west-1"]),
    }
  ]

  // origin DC steps. Prepare one origin DC at a time. Used by all origin machine class configs.
  origin_dc_steps = [for dc in module.datacenters.origin_datacenters_in_commission :
    { name                              = "origin-rollout-${dc}",
      pops                              = [dc],
      deploy_config_name                = "CodeDeployDefault.OneAtATime",
      max_concurrent_deployment_actions = 1,
    }
  ]

  // Structured, exported metadata to facilitate codedeploy/pipeline module consumption.
  // pipeline_name is a separate field from application-name
  deploy_account_map = {
    twitch-vidcs-consul-master-prod = {
      account_id       = local.deploy_acc_metadata["twitch-vidcs-consul-master-prod"]["account_id"],
      application_name = local.deploy_acc_metadata["twitch-vidcs-consul-master-prod"]["name"],
      machine_classes  = local.deploy_acc_metadata["twitch-vidcs-consul-master-prod"]["machine_classes"],
      pipeline_name    = local.deploy_acc_metadata["twitch-vidcs-consul-master-prod"]["name"],
      pipeline_stages = [for step in local.origin_dc_steps :
        {
          machine_classes                   = local.deploy_acc_metadata["twitch-vidcs-consul-master-prod"]["machine_classes"]
          stage_description                 = step["name"]
          pops                              = step["pops"]
          environment                       = "production"
          approval_config                   = module.default_behavior.no_approval_config
          codedeploy_account_id             = local.deploy_acc_metadata["twitch-vidcs-consul-master-prod"]["account_id"]
          minimum_healthy_hosts_percent     = 100
          max_concurrent_deployment_actions = step["max_concurrent_deployment_actions"]
          deployment_actions                = module.default_behavior.no_chunk_deployment_actions
          codedeploy_deployment_config_name = step["deploy_config_name"]
        }
      ],
    },
    twitch-vidcs-dns-auth-prod = {
      account_id       = local.deploy_acc_metadata["twitch-vidcs-dns-auth-prod"]["account_id"],
      application_name = local.deploy_acc_metadata["twitch-vidcs-dns-auth-prod"]["name"],
      machine_classes  = local.deploy_acc_metadata["twitch-vidcs-dns-auth-prod"]["machine_classes"],
      pipeline_name    = local.deploy_acc_metadata["twitch-vidcs-dns-auth-prod"]["name"],
      pipeline_stages = [for step in local.origin_dc_steps :
        {
          machine_classes                   = local.deploy_acc_metadata["twitch-vidcs-dns-auth-prod"]["machine_classes"]
          stage_description                 = step["name"]
          pops                              = step["pops"]
          environment                       = "production"
          approval_config                   = module.default_behavior.no_approval_config
          codedeploy_account_id             = local.deploy_acc_metadata["twitch-vidcs-dns-auth-prod"]["account_id"]
          minimum_healthy_hosts_percent     = 100
          max_concurrent_deployment_actions = step["max_concurrent_deployment_actions"]
          deployment_actions                = module.default_behavior.no_chunk_deployment_actions
          codedeploy_deployment_config_name = step["deploy_config_name"]
        }
      ],
    },
    twitch-vidcs-dns-resolver-prod = {
      account_id       = local.deploy_acc_metadata["twitch-vidcs-dns-resolver-prod"]["account_id"],
      application_name = local.deploy_acc_metadata["twitch-vidcs-dns-resolver-prod"]["name"],
      machine_classes  = local.deploy_acc_metadata["twitch-vidcs-dns-resolver-prod"]["machine_classes"],
      pipeline_name    = local.deploy_acc_metadata["twitch-vidcs-dns-resolver-prod"]["name"],
      pipeline_stages = [for step in local.origin_dc_steps :
        {
          machine_classes                   = local.deploy_acc_metadata["twitch-vidcs-dns-resolver-prod"]["machine_classes"]
          stage_description                 = step["name"]
          pops                              = step["pops"]
          environment                       = "production"
          approval_config                   = module.default_behavior.no_approval_config
          codedeploy_account_id             = local.deploy_acc_metadata["twitch-vidcs-dns-resolver-prod"]["account_id"]
          minimum_healthy_hosts_percent     = 100
          max_concurrent_deployment_actions = step["max_concurrent_deployment_actions"]
          deployment_actions                = module.default_behavior.no_chunk_deployment_actions
          codedeploy_deployment_config_name = step["deploy_config_name"]
        }
      ],
    },
    twitch-vidcs-pop-masters-prod = {
      account_id       = local.deploy_acc_metadata["twitch-vidcs-pop-masters-prod"]["account_id"],
      application_name = local.deploy_acc_metadata["twitch-vidcs-pop-masters-prod"]["name"],
      machine_classes  = local.deploy_acc_metadata["twitch-vidcs-pop-masters-prod"]["machine_classes"],
      pipeline_name    = local.deploy_acc_metadata["twitch-vidcs-pop-masters-prod"]["name"],
      pipeline_stages = [for step in concat(local.edge_dc_steps, local.origin_dc_steps) :
        {
          machine_classes                   = local.deploy_acc_metadata["twitch-vidcs-pop-masters-prod"]["machine_classes"]
          stage_description                 = step["name"]
          pops                              = step["pops"]
          environment                       = "production"
          approval_config                   = module.default_behavior.no_approval_config
          codedeploy_account_id             = local.deploy_acc_metadata["twitch-vidcs-pop-masters-prod"]["account_id"]
          minimum_healthy_hosts_percent     = 100
          max_concurrent_deployment_actions = step["max_concurrent_deployment_actions"]
          deployment_actions                = module.default_behavior.no_chunk_deployment_actions
          codedeploy_deployment_config_name = step["deploy_config_name"]
        }
      ],
    },
    twitch-vidcs-squid-proxy-prod = {
      account_id       = local.deploy_acc_metadata["twitch-vidcs-squid-proxy-prod"]["account_id"],
      application_name = local.deploy_acc_metadata["twitch-vidcs-squid-proxy-prod"]["name"],
      machine_classes  = local.deploy_acc_metadata["twitch-vidcs-squid-proxy-prod"]["machine_classes"],
      pipeline_name    = local.deploy_acc_metadata["twitch-vidcs-squid-proxy-prod"]["name"],
      pipeline_stages = [for step in local.origin_dc_steps :
        {
          machine_classes                   = local.deploy_acc_metadata["twitch-vidcs-squid-proxy-prod"]["machine_classes"]
          stage_description                 = step["name"]
          pops                              = step["pops"]
          environment                       = "production"
          approval_config                   = module.default_behavior.no_approval_config
          codedeploy_account_id             = local.deploy_acc_metadata["twitch-vidcs-squid-proxy-prod"]["account_id"]
          minimum_healthy_hosts_percent     = 100
          max_concurrent_deployment_actions = step["max_concurrent_deployment_actions"]
          deployment_actions                = module.default_behavior.no_chunk_deployment_actions
          codedeploy_deployment_config_name = step["deploy_config_name"]
        }
      ],
    },
  }
}
