#!/usr/bin/env bash

set -eo pipefail

export AWS_STS_REGIONAL_ENDPOINTS=regional

root_dir=$(git rev-parse --show-toplevel)

usage() {
    mapfile -t valid_accounts < <(find "${root_dir}/accounts/deploy" -type f | sed -r 's/^.*\/(.*)\/(.*\..*)$/\1/')

    cat <<EOF
Usage: $0 -a <account-alias> <command> [args ...]
  <account-alias>
    The AWS account alias for the deploy account.
    Valid accounts:
      $(printf '%s\n      ' "${valid_accounts[@]}")
  <command> [args ...]
    In general, these are command and args that are passed to terraform binary as
    arguments
    E.g.
      $0 -a <account-alias> plan
      gets passed to terraform as
      terraform plan
EOF
    exit 1
}

tf_init() {
    shift
    terraform init "$@" -reconfigure -backend-config=<(cat <<EOF
bucket = "${TF_S3_STATE_BUCKET}"
EOF
)
    terraform get -update
}

validate_account() {
    if ! [[ "$AWS_ACCOUNT_ID" == "$(aws sts get-caller-identity | jq -r .Account)" ]]; then
        echo
        echo "Your current credentials do not match account '${AWS_ACCOUNT_ID}' that manages '${account}' domain."
        echo "Please double check that you are using the correct credentials for the correct account."
        echo
        usage
    fi
}

command -v tfenv &> /dev/null || {
    echo "tfenv required but missing" && usage
}

[ "$1" == "-a" ] || usage

account="$2"
shift 2

account_conf_path="${root_dir}/accounts/deploy/${account}/"

# Source account vars
if [[ -f "${account_conf_path}/account.conf" ]]; then
    # shellcheck disable=SC1090
    . "${account_conf_path}/account.conf"
else
    echo "ERROR: No account.conf found under ${account_conf_path}. Please create and set TF_S3_STATE_BUCKET appropriately."
    usage
fi

case "$1" in
    init)
        tf_init "$@"
        ;;
    plan|apply)
        tf_init "$@"
        exec terraform "$@" \
            -var="account_alias=${account}"
        ;;
    destroy)
        exec terraform "$@" \
            -var="account_alias=${account}"
        ;;
    *)
        exec terraform "$@"
        ;;
esac
