terraform {
  backend "s3" {
  }
}

module "variables" {
  source = "./vendor/modules/twitch-video-ops-defaults/current"
  region = var.region
}

locals {
  non_region_zone_name = replace(module.variables.svc_puppet_r53_zone_name, "${var.region}.", "")
}

module "puppet_master_asg" {
  source = "./vendor/modules/asg"

  service         = var.service
  region          = var.region
  env             = var.env
  route53_zone_id = module.variables.svc_puppet_r53_zone_id
  vpc_id          = module.variables.vpc_id

  user_data = templatefile("${path.module}/templates/user_data.sh.tmpl", {
    non_region_zone_name = local.non_region_zone_name
    zone_name            = module.variables.svc_puppet_r53_zone_name
    file_system_id       = module.puppet-env-backend.file_system_id
    region               = var.region
  })

  owner              = "video-coreservices@twitch.tv"
  puppet_team        = "vidcs"
  min_size           = var.instance_count
  max_size           = var.instance_count
  security_group_ids = [module.variables.sg_twitch_subnet_id]
  subnet_ids         = module.variables.private_subnet_ids
  instance_type      = var.instance_type
  puppet_server      = "video-puppet.${local.non_region_zone_name}"
  puppet_ca_server   = "video-puppet-ca.${local.non_region_zone_name}"

  target_group_arns = [aws_lb_target_group.puppet_tg.arn]

  ebs_block_device = [{
    device_name = "/dev/xvdf"
    volume_type = "gp2"
    volume_size = var.ebs_volume_size
  }]

  tags = {
    Service = "puppet"
    Source  = "video-coreservices/tf-io-puppet"
  }
}

# Sandstorm access to webhook keys
resource "aws_iam_policy" "sandstorm_access" {
  name        = "${var.service}-code-sandstorm"
  description = "Access to video-puppet/puppet-monolith"

  policy = data.aws_iam_policy_document.sandstorm_access.json
}

data "aws_iam_policy_document" "sandstorm_access" {
  version = "2012-10-17"

  statement {
    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/vidops-video-puppet-master",
    ]

    effect = "Allow"
  }
}

resource "aws_iam_role_policy_attachment" "sandstorm_access_attachment_asg" {
  role       = module.puppet_master_asg.iam_role_name
  policy_arn = aws_iam_policy.sandstorm_access.arn
}

# Cloudwatch access for video-cloudwatch-agent
resource "aws_iam_policy" "cloudwatch_access" {
  name        = "${var.service}-video-cloudwatch-access"
  description = "Access to writing to central systems metrics/logs"

  policy = data.aws_iam_policy_document.cloudwatch_access.json
}

data "aws_iam_policy_document" "cloudwatch_access" {
  version = "2012-10-17"

  statement {
    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "arn:aws:iam::277437172290:role/CloudwatchAgentMetrics",
      "arn:aws:iam::277437172290:role/CloudwatchAgentLogs"
    ]

    effect = "Allow"
  }
}

resource "aws_iam_role_policy_attachment" "cloudwatch_access_attachment_asg" {
  role       = module.puppet_master_asg.iam_role_name
  policy_arn = aws_iam_policy.cloudwatch_access.arn
}
