#!/usr/bin/env bash

set -eo pipefail

usage() {
    cat <<EOF
Usage: $0 -s <service> -e <env> -r <region> <command> [args]

<service>
  The name of the service. E.g. puppet

<region>
  This should be a valid AWS region. E.g. us-west-2

<env>
  This should be either dev, stage, or prod

<command> [args]
  These are command and args that are passed to terraform binary as arguments.

  E.g.
    $0 -s <service> -e <env> -r <region> plan

    gets passed to terraform as

    terraform plan
EOF
    exit 1
}

requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && usage )
    [[ -f "$varfile" ]] || ( echo "$varfile file does not exist" && usage )
}

tf_init() {
    shift
    terraform init "$@" -reconfigure -backend-config=<(cat <<EOF
key    = "delegations/${region}.${env}.${service}.tfstate"
bucket = "${service}-live-video-a2z-dns-tf-state"
EOF
)
}

while getopts 'hs:e:r:' option; do
    case "$option" in
        s)
            service="$OPTARG"
            ;;
        e)
            env="$OPTARG"
            ;;
        r)
            region="$OPTARG"
            ;;
        h)
            usage
            ;;
        *)
            ;;
    esac
done

# Make sure everything is set
[[ -z "$service" || -z "$env" || -z "$region" ]] && usage

# in case this script is not run in it's root directory
realpath="$(realpath "$0" | sed 's|\(.*\)/.*|\1|')"
varfile="${realpath}/../../conf/${service}/${env}.${region}.tfvars"

# Shift arguments
shift 6

requirements

# Terraform commands
case "$1" in
    init)
        tf_init "$@"
        ;;
    plan|apply)
        tf_init "$@"
        exec terraform "$@" \
            -var-file="${varfile}" \
            -var="env=${env}" \
            -var="region=${region}" \
            -var="service=${service}"
        ;;
    *)
        exec terraform "$@"
        ;;
esac
