#!/usr/bin/env bash

set -eo pipefail

usage() {
    cat <<EOF
Sets up lvs-dns-delegation IAM role that can be assumed to create and delegate subdomain hosted zones.

Usage: $0 <command> [args]

<command> [args]
  These are command and args that are passed to terraform binary as arguments.

  E.g.
    $0 plan

    gets passed to terraform as

    terraform plan
EOF
    return 1
}

requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && usage )
    command -v aws &> /dev/null || ( echo "aws required but missing" && usage )
}

check_existing_role() {
    # Check valid credentials
    aws sts get-caller-identity 1>/dev/null || usage

    # Check existence of role and prompt to import/continue
    aws iam get-role --role-name lvs-dns-delegation &> /dev/null || return 0
    cat <<EOF
Role 'lvs-dns-delegation' has already been created in this AWS account!

Do not continue and run the following if want to import existing role and policies:

$0 import module.iam.aws_iam_role.lvs_dns_delegation lvs-dns-delegation
$0 import module.iam.aws_iam_role_policy_attachment.readonly lvs-dns-delegation/arn:aws:iam::aws:policy/AmazonRoute53ReadOnlyAccess
$0 import module.iam.aws_iam_role_policy.route53_create_hosted_zone[0] lvs-dns-delegation:Route53CreateHostedZone
EOF
    echo -n "Do you still want to continue? [N|y]: "
    read -r answer

    [[ "$answer" != "y" ]] && return 1

    # Check to continue using existing state file
    [[ -f terraform.tfstate ]] || return 0
    cat <<EOF

terraform.tfstate state file found!

Do not continue and run the following if you do not want to use the existing state file.

rm -f terraform.tfstate
EOF
    echo -n "Do you wish to use this terraform.tfstate state file? [N|y]: "
    read -r answer
    [[ "$answer" != "y" ]] && return 1 || return 0
}

[[ "$#" -lt 1 || "$1" == "-h" ]] && usage

requirements

case "$1" in
    import|output)
        exec terraform "$@"
        ;;
    *)
        check_existing_role
        exec terraform "$@"
        ;;
esac
