#!/usr/bin/env bash

set -eo pipefail

export AWS_STS_REGIONAL_ENDPOINTS=regional
export AWS_SDK_LOAD_CONFIG=true

usage() {
    cat <<EOF >&2
Usage:
  $0 -e <env> -r <region> <command> [args...]

  <region>
    This should be a valid AWS region. E.g. us-west-2

  <env>
    This should be either dev, staging, or prod

  <command> [args ...]
    In general, these are command and args that are passed to terraform binary as
    arguments

    E.g.
      $0 plan
      gets passed to terraform as
      terraform plan
EOF
    exit 1
}

err() {
    echo "ERROR: $*" >&2
}

requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && usage )
}

tf_init() {
    shift
    terraform init "$@" -reconfigure -backend-config=<(cat <<EOF
key            = "$TF_KEY"
bucket         = "$TF_BUCKET"
dynamodb_table = "$TF_DYNAMODB_TABLE"
region         = "$region"
EOF
)
}

validate_account() {
    if ! [[ "$AWS_ACCOUNT_ID" == "$(aws sts get-caller-identity | jq -r .Account)" ]]; then
        err "$(cat <<EOF

  Your current credentials do not match account '${AWS_ACCOUNT_ID}' that manages this environment and region
  Please double check that you are using the correct credentials for the correct account.
EOF
)"
        usage
    fi
}

while getopts "hs:e:r:" option; do
  case "$option" in
    e)
      env="$OPTARG"
      ;;
    r)
      region="$OPTARG"
      ;;
    h)
      usage
      ;;
    *)
      ;;
  esac
done

# Make sure everything is set
[[ -z "$env" || -z "$region" ]] && usage

# Source env configs
if [[ -f "conf/${env}/${region}/backend.conf" ]]; then
    # shellcheck source=/dev/null
    . "conf/${env}/${region}/backend.conf"
else
    err "conf/${env}/${region}/backend.conf is missing"
    usage
fi

requirements
validate_account

shift $((OPTIND-1))

case "$1" in
    init)
        tf_init "$@"
        ;;
    plan|apply)
        tf_init "$@"
        exec terraform "$@" \
            -var="region=${region}" \
            -var="env=${env}" \
            -var-file="conf/${env}/${region}/variables.tfvars"
        ;;
    *)
        exec terraform "$@"
        ;;
esac
