# Role used for account-level ECS services.
resource "aws_iam_role" "ecs_task_role" {
  name = "ecs_task_role_${local.account}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "cloudwatch_metrics_policy" {
  name   = "${local.account}_cloudwatch_metrics_policy"
  role   = aws_iam_role.ecs_task_role.id
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Action": [
        "cloudwatch:PutMetricData"
      ],
      "Resource": "*"
    }
  ]
}
EOF
}

module "core_account" {
  source = "git::git+ssh://git@git.xarth.tv/video-coreservices/tf-mod-voncount-core-account?ref=v1.0.0"

  environment = var.env
  team        = local.account
  xray_role   = "*"
  # write to all the xray's
}

# Auto populate max instances based on desired instances
locals {
  statsd_proxy_asg_instances = {
    type    = "c5n.2xlarge"
    desired = var.statsd_proxy_asg_instances
    max     = var.statsd_proxy_asg_instances * 2
  }

  statsite_cluster_size = {
    desired = var.statsite_cluster_size
    max     = var.statsite_cluster_size * 2
  }
}

module "statsd_proxy" {
  source = "git@git.xarth.tv:video-coreservices/go-statsd-proxy.git//terraform?ref=v0.5.3"

  environment = local.account

  vpc_id                 = var.vpc_id
  vpc_security_group_ids = var.security_groups
  subnet_ids             = var.subnets

  stats_cluster_arn  = module.statsite.ecs_cluster_arn
  stats_cluster_name = module.statsite.ecs_cluster_name
  stats_service_name = module.statsite.ecs_service_name

  image_tag = var.statsd_proxy_image_tag

  asg_instances = local.statsd_proxy_asg_instances

  # The ECS task reservations depend on the size of the instance type.
  # Here, we want the tasks to take up an entire instance, but allow some room for the OS and ECS agent.
  reservation = {
    memory_mib = floor(21 * 1024 * 0.9) # Reserve 90% of c5n.2xlarge's 21 gibibyte capacity.
    cpu_units  = floor(8 * 1024 * 0.9)  # Reserve 90% of c5n.2xlarge's 8 vCPU capacity, in CPU units (1 vCPU = 1024 units).
  }

  pagerduty_sns_topic_arns = var.pagerduty_sns_topic_arns
}


resource "aws_cloudwatch_log_group" "statsite" {
  name              = "statsite_logs"
  retention_in_days = 30

  tags = {
    environment = local.account
  }
}

module "statsite" {
  source = "./modules/statsite"

  cluster_size = local.statsite_cluster_size

  vpc_id          = var.vpc_id
  subnets         = var.subnets
  security_groups = var.security_groups

  instance_type      = "c5.xlarge"
  tasks_per_instance = 4
  memory             = "512"

  environment  = var.env
  region       = var.region
  cluster_name = "statsite"
  image_url    = "${aws_ecr_repository.statsite.repository_url}:${var.statsite_image_tag}"

  owner_email = var.owner

  service_iam_role              = module.core_account.service_iam_role
  container_instance_profile_id = module.core_account.container_instance_profile_id

  ecs_task_role_arn = aws_iam_role.ecs_task_role.arn

  log_group = aws_cloudwatch_log_group.statsite.name

  graphite_host  = var.graphite_url
  graphite_port  = 2003
  flush_interval = 10

  alarm_threshold          = 300000
  pagerduty_sns_topic_arns = var.pagerduty_sns_topic_arns
}
