# ECS Module
#
# Uses the "ecsasg" module in to set up the EC2 auto-scaling group for ECS.
# The actual ECS cluster is created here.

data "aws_iam_account_alias" "current" {}

data "aws_caller_identity" "current" {}

### Logging

resource "aws_iam_role_policy_attachment" "allow_send_access_logs_policy" {
  role       = var.ecs_host_iam_id
  policy_arn = "arn:aws:iam::aws:policy/AWSOpsWorksCloudWatchLogs"
}

resource "aws_cloudwatch_log_group" "ecs_agent_logs" {
  name              = "${var.cluster_name}-ecs-agent-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app     = var.app_name
    cluster = var.cluster_name
    env     = var.environment
  }
}

resource "aws_cloudwatch_log_group" "ecs_init_logs" {
  name              = "${var.cluster_name}-ecs-init-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app     = var.app_name
    cluster = var.cluster_name
    env     = var.environment
  }
}

resource "aws_cloudwatch_log_group" "ssm_logs" {
  name              = "${var.cluster_name}-ssm-ec2-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app     = var.app_name
    cluster = var.cluster_name
    env     = var.environment
  }
}

### Container instance AutoScaling Group (ASG).

# Define how our upcoming ECS cluster will get autoscaled & how the cluster's EC2's are deployed and configured
module "ecs_asg" {
  source = "../ecsasg"

  cluster_name                  = var.cluster_name
  task_cleanup_wait_duration    = var.task_cleanup_wait_duration
  task_shutdown_wait_duration   = var.task_shutdown_wait_duration
  container_instance_profile_id = var.ecs_host_iam_id
  desired_capacity              = var.cluster_size["desired"]
  max_size                      = var.cluster_size["max"]
  ami                           = var.ami
  environment                   = var.environment
  instance_type                 = var.ec2_instance_type
  name                          = var.cluster_name
  owner                         = var.owner_email

  ecs_agent_loggroup = aws_cloudwatch_log_group.ecs_agent_logs.name
  ecs_init_loggroup  = aws_cloudwatch_log_group.ecs_agent_logs.name
  ssm_agent_loggroup = aws_cloudwatch_log_group.ssm_logs.name
  team               = var.team_name

  security_groups = var.security_groups

  # Subnets to which ASG will add new ec2's. Yeah the var name is weird
  vpc_zone_identifier = join(",", var.subnets)
}

### ECS Cluster

# This cluster capacity provider enables the ECS cluster to add container instances as needed.
resource "aws_ecs_capacity_provider" "main" {
  name = "provider_${var.team_name}_${var.cluster_name}"

  auto_scaling_group_provider {
    auto_scaling_group_arn         = module.ecs_asg.arn
    managed_termination_protection = "ENABLED"

    managed_scaling {
      status          = "ENABLED"
      target_capacity = 100
    }
  }
}

resource "aws_ecs_cluster" "main" {
  name = var.cluster_name

  capacity_providers = [aws_ecs_capacity_provider.main.name]
  default_capacity_provider_strategy {
    capacity_provider = aws_ecs_capacity_provider.main.name
  }

  setting {
    name  = "containerInsights"
    value = "enabled"
  }
}

### ECS Service

resource "aws_iam_role" "ecs_task_execution_role" {
  name = "ecs_task_execution_${var.app_name}_${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "ecs_task_execution_role" {
  role       = aws_iam_role.ecs_task_execution_role.id
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_ecs_task_definition" "app" {
  family                = var.cluster_name
  network_mode          = var.task_network_mode
  execution_role_arn    = aws_iam_role.ecs_task_execution_role.arn
  task_role_arn         = var.ecs_task_role
  container_definitions = var.container_definitions
}

resource "aws_ecs_service" "main" {
  name            = var.cluster_name
  cluster         = aws_ecs_cluster.main.id
  task_definition = aws_ecs_task_definition.app.arn
  desired_count   = var.app_count
  iam_role        = length(var.target_group_ids) > 0 ? var.service_iam_role : ""

  enable_ecs_managed_tags = true

  capacity_provider_strategy {
    capacity_provider = aws_ecs_capacity_provider.main.name
    weight            = 1
  }

  # Uncomment should autoscaling comes in useful later
  # lifecycle {
  #   ignore_changes = [
  #     desired_count # Don't change autoscaled counts when applying changes.
  #   ]
  # }

  ordered_placement_strategy {
    type  = var.placement_strategy_type
    field = var.placement_strategy_field
  }

  dynamic "load_balancer" {
    for_each = var.target_group_ids
    iterator = target_group_id
    content {
      target_group_arn = target_group_id.value
      container_name   = var.cluster_name
      container_port   = var.app_port
    }
  }
}
