locals {
  app_name = "statsite"
  ulimit   = 10000
}

module "ecs" {
  source = "../ecs"

  app_name     = local.app_name
  environment  = var.environment
  cluster_name = var.cluster_name

  ec2_instance_type = var.instance_type
  app_count         = var.tasks_per_instance * var.cluster_size.desired
  cluster_size      = var.cluster_size
  use_dnsmasq       = false
  # statsite is single threaded, so we must run 1 task per CPU. This assumes |tasks| = |vCPUs|
  placement_strategy_type  = "binpack"
  placement_strategy_field = "cpu"

  security_groups = join(",", var.security_groups)
  subnets         = var.subnets

  ecs_host_iam_id = var.container_instance_profile_id

  app_port              = var.app_port
  container_definitions = <<DEFINITION
[
  {
    "name": "${var.cluster_name}",
    "image": "${var.image_url}",
    "cpu": 1024,
    "memory": ${var.memory},
    "essential": true,
      "portMappings": [
        {
          "protocol": "tcp",
          "containerPort": ${var.app_port}
        },
        {
          "protocol": "udp",
          "containerPort": ${var.app_port}
        }
      ],
      "environment": [
        {
          "name": "TCP_PORT",
          "value": "${var.app_port}"
        },
        {
          "name": "UDP_PORT",
          "value": "${var.app_port}"
        },
        {
          "name": "APP_NAME",
          "value": "${var.app_name}"
        },
        {
          "name": "STAGE",
          "value": "${var.stage}"
        },
        {
          "name": "AWS_REGION",
          "value": "${var.region}"
        },
        {
          "name": "FLUSH_INTERVAL",
          "value": "${var.flush_interval}"
        },
        {
          "name": "GRAPHITE_HOST",
          "value": "${var.graphite_host}"
        },
        {
          "name": "GRAPHITE_PORT",
          "value": "${var.graphite_port}"
        }
      ],
      "logConfiguration": {
        "logDriver": "awslogs",
        "options": {
          "awslogs-group": "${var.log_group}",
          "awslogs-region": "${var.region}",
          "awslogs-stream-prefix": "${local.app_name}"
        }
      },
    "ulimits": [
      {
        "name": "nofile",
        "softLimit": ${local.ulimit},
        "hardLimit": ${local.ulimit}
      }
    ]
  }
]
DEFINITION

  ecs_task_role = var.ecs_task_role_arn

  owner_email      = var.owner_email
  service_iam_role = var.service_iam_role
}

resource "aws_cloudwatch_metric_alarm" "metrics_written" {
  count               = var.alarm_threshold == 0 ? 0 : 1
  alarm_name          = "statsite_metrics_written"
  alarm_description   = "Alarm when statsite is not emitting metrics"
  comparison_operator = "LessThanThreshold"
  // metrics-written is reported by plumbago, the custom flusher we use with statsite
  namespace   = "plumbago"
  metric_name = "metrics-written"
  statistic   = "Sum"
  dimensions = {
    region = var.region
  }

  // 10 minutes is so that we don't alert when deploying go-statsd-proxy and statsite (~5min)
  evaluation_periods = "10"
  period             = "60"
  threshold          = var.alarm_threshold
  alarm_actions      = var.pagerduty_sns_topic_arns
  ok_actions         = var.pagerduty_sns_topic_arns
  treat_missing_data = "missing"
}

resource "aws_cloudwatch_metric_alarm" "cpu_utilization" {
  alarm_name          = "${var.cluster_name}_cpu_utization"
  alarm_description   = "Alarm when ${var.cluster_name}'s cpu utilization is > 80% for 5 minute."
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "CPUUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  alarm_actions      = var.pagerduty_sns_topic_arns
  ok_actions         = var.pagerduty_sns_topic_arns
  treat_missing_data = "missing"
}

resource "aws_cloudwatch_metric_alarm" "memory_utilization" {
  alarm_name          = "${var.cluster_name}_memory_utilization"
  alarm_description   = "Alarm when ${var.cluster_name}'s memory utilization is > 80% for 5 minute."
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "MemoryUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  alarm_actions      = var.pagerduty_sns_topic_arns
  ok_actions         = var.pagerduty_sns_topic_arns
  treat_missing_data = "missing"
}
