#!/usr/bin/env bash

set -eo pipefail

ENV="staging"
REGION="us-west-2"

declare -A account_map
account_map["prod:us-west-2"]=733068333177
account_map["prod:us-east-2"]=618029135405
account_map["staging:us-west-2"]=410778197068


while getopts ":e:r:" opt; do
  case ${opt} in
    e )
      ENV=$OPTARG
      ;;
    r )
      REGION=$OPTARG
      ;;
    \? )
      echo "Usage: $0 [ -e prod|staging ][ -r <aws_region> ] [init|plan|apply]" 1>&2
      exit 1
      ;;
  esac
done


requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && exit 1 )
    command -v jq &> /dev/null || ( echo "jq required but missing" && exit 1 )
}

check_valid_account() {
  current_acc_id=$(aws sts get-caller-identity | jq -r '.Account | tonumber')
  acc_key="${ENV}:${REGION}"
  if [[ ! -v "account_map[${acc_key}]" ]] ; then
    echo "deploy pairing ${ENV} ${REGION} does not exist. If adding a new region, add it as a legal option in deploy.sh." && exit 1
  fi
  if [[ "account_map[${acc_key}]" -ne "${current_acc_id}" ]] ; then
    echo "You are using credentials for the account ${current_acc_id} but ${account_map[${acc_key}]} is required." && exit 1
  fi
}

tf_init() {
    shift
    terraform init "$@" -reconfigure -backend-config="./conf/${ENV}/${REGION}/backend.tfvars"
    terraform get -update
}

requirements
check_valid_account

CMD=${@:$OPTIND:1}

shift $((OPTIND-1))

case "$CMD" in
    init)
        tf_init "$@"
        ;;
    plan|apply|destroy)
        tf_init "$@"
        exec terraform "$@" \
            -var-file="./conf/${ENV}/${REGION}/variables.tfvars"
        ;;
    *)
        exec terraform "$@"
        ;;
esac
