###########################################################
# Manage twitch-pop-warp-dns permissions for DNS changes
# For console/sts access, permit posix groups via Isengard
#   https://isengard.amazon.com/roles/804916306882
###########################################################

locals {
  role_name = "twitch-pop-warp-dns"
}

resource "aws_iam_policy" "pop_warp_crud_policy" {
  name        = "twitch-pop-warp-dns-route53"
  description = "Permit DNS record changes on all Hosted Zones with the exception of warp.live-video.net hosted zone"

  policy = templatefile("${path.module}/templates/twitch-pop-warp-dns-policy.json.tmpl", {
    restricted_zone = module.primary.primary.zone.id
  })
}

resource "aws_iam_role_policy_attachment" "pop_dns_crud_attachment" {
  role       = local.role_name
  policy_arn = aws_iam_policy.pop_warp_crud_policy.arn
}

resource "aws_iam_role_policy_attachment" "pop_dns_readonly_attachment" {
  role       = local.role_name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}

###########################################################
# Permit route53 service to log to any /aws/route53 log group
###########################################################

# cloudWatch log resource policy to allow Route53 to write logs
# to any log group under /aws/route53/warp-live-video-net-query-logs
data "aws_iam_policy_document" "r53_query_logging_policy" {
  statement {
    actions = [
      "logs:CreateLogStream",
      "logs:PutLogEvents",
    ]
    resources = ["arn:aws:logs:us-east-1:${data.aws_caller_identity.current.account_id}:log-group:${local.r53_query_log_group}*"]

    principals {
      identifiers = ["route53.amazonaws.com"]
      type        = "Service"
    }
  }
}

resource "aws_cloudwatch_log_resource_policy" "r53_query_logging_policy" {
  policy_document = data.aws_iam_policy_document.r53_query_logging_policy.json
  policy_name     = "route53-query-logging-policy"
}

################################################
# Permit prod crud_updater lambda to operate changes
################################################
data "aws_iam_policy_document" "crud_updater" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        # twitch-warp-dns+edgeupdate-prod-us-east-2@amazon.com
        "arn:aws:iam::618029135405:role/warp-dns-updater-lambda-role",
        # twitch-warp-dns+edgeupdate-prod-us-west-2@amazon.com
        "arn:aws:iam::733068333177:role/warp-dns-updater-lambda-role",
      ]
    }
  }
}

resource "aws_iam_role" "crud_updater" {
  name               = "warp-dns-crud-lambda"
  assume_role_policy = data.aws_iam_policy_document.crud_updater.json
}

resource "aws_iam_role_policy_attachment" "pop_dns_crud_updater_attachment" {
  role       = aws_iam_role.crud_updater.name
  policy_arn = aws_iam_policy.pop_warp_crud_policy.arn
}

resource "aws_iam_role_policy_attachment" "pop_dns_crud_updater_readonly_attachment" {
  role       = aws_iam_role.crud_updater.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}

################################################
# Permit management of pop subdomain delegation
# from video-coreservices/tf-io-dns-pop-delegation
################################################

module "pop_delegation" {
  source           = "git@git.xarth.tv:video-coreservices/tf-mod-live-video-dns-delegation.git//modules/pop_delegation_role?ref=v1.0.10"
  primary_zone_arn = module.primary.primary.zone.id
}

################################################
# Permit warp staging account to perform R53 operations on the staging subdomain.
################################################
data "aws_iam_policy_document" "warp_staging_subdomain" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"
      identifiers = [
        # twitch-warp-dns+edgeupdate-staging-us-west-2@amazon.com
        "arn:aws:iam::410778197068:role/warp-dns-updater-lambda-role",
      ]
    }
  }
}

resource "aws_iam_policy" "warp_staging_subdomain" {
  name        = "warp-staging-dns"
  description = "Permit DNS record changes on specific Hosted Zones for the staging warp DNS updater account."

  policy = templatefile("${path.module}/templates/twitch-warp-staging-dns-policy-limited.json.tmpl", {
    zone = "Z0381546E021MA7OII1Q"
  })
}

resource "aws_iam_role" "warp_staging_subdomain" {
  name               = "warp-staging-subdomain-crud-allow"
  assume_role_policy = data.aws_iam_policy_document.warp_staging_subdomain.json
}

resource "aws_iam_role_policy_attachment" "warp_staging_subdomain_attachment" {
  role       = aws_iam_role.warp_staging_subdomain.name
  policy_arn = aws_iam_policy.warp_staging_subdomain.arn
}

resource "aws_iam_role_policy_attachment" "warp_staging_subdomain_readonly_attachment" {
  role       = aws_iam_role.warp_staging_subdomain.name
  policy_arn = "arn:aws:iam::aws:policy/ReadOnlyAccess"
}
