#!/usr/bin/env bash

set -eo pipefail

usage() {
  cat <<EOF >&2
Usage:
  $0 -s <service> -e <env> -r <region> <command> [args...]

  <service>
    The name of the service. E.g. puppet
  <region>
    This should be a valid AWS region. E.g. us-west-2
  <env>
    This should be either dev, staging, or prod
  <command> [args ...]
    In general, these are command and args that are passed to terraform binary as
    arguments
    E.g.
      $0 plan
      gets passed to terraform as
      terraform plan

This setups the minimum required S3 bucket and dynamodb locks to run terraform on this account.
EOF
  exit 1
}

requirements() {
    command -v tfenv &> /dev/null || ( echo "tfenv required but missing" && usage )
    command -v jq &> /dev/null || ( echo "jq required but missing" && usage )
}

tf_init() {
    shift
    terraform init "$@" -reconfigure
}

while getopts "hs:e:r:" option; do
  case "$option" in
    s)
      service="$OPTARG"
      ;;
    e)
      env="$OPTARG"
      ;;
    r)
      region="$OPTARG"
      ;;
    h)
      usage
      ;;
    *)
      ;;
  esac
done

# Make sure everything is set
[[ -z "$service" || -z "$env" || -z "$region" ]] && usage

shift $((OPTIND-1))

requirements

account_id="$(aws sts get-caller-identity | jq -r .Account)"

case "$1" in
    init)
        tf_init "$@"
        ;;
    plan|apply)
        tf_init "$@"
        exec terraform "$@" \
          -var="service=${service}" \
          -var="region=${region}" \
          -var="env=${env}" \
          -state="${account_id}-${service}-${region}-${env}.tfstate"
        ;;
    *)
        exec terraform "$@"
        ;;
esac

