locals {
  datacenters = var.datacenters

  aws_regions = sort(distinct([for dc, attr in local.datacenters : attr.aws_region]))

  origin_dcs                     = [for dc, attr in local.datacenters : dc if attr.origin == true]
  origin_dcs_incomm              = [for dc, attr in local.datacenters : dc if attr.origin == true && attr.in_commission == true]
  origin_dcs_decomm              = [for dc, attr in local.datacenters : dc if attr.origin == true && attr.in_commission == false]
  origin_dc_to_region_map        = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.origin == true }
  region_to_origin_dc_map        = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.origin == true }
  origin_dc_to_region_map_incomm = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.origin == true && attr.in_commission == true }
  origin_dc_to_region_map_decomm = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.origin == true && attr.in_commission == false }
  region_to_origin_dc_map_incomm = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.origin == true && attr.in_commission == true }
  region_to_origin_dc_map_decomm = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.origin == true && attr.in_commission == false }

  edge_dcs                     = [for dc, attr in local.datacenters : dc if attr.edge == true]
  edge_dcs_incomm              = [for dc, attr in local.datacenters : dc if attr.edge == true && attr.in_commission == true]
  edge_dcs_decomm              = [for dc, attr in local.datacenters : dc if attr.edge == true && attr.in_commission == false]
  edge_dc_to_region_map        = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.edge == true }
  region_to_edge_dc_map        = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.edge == true }
  edge_dc_to_region_map_incomm = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.edge == true && attr.in_commission == true }
  edge_dc_to_region_map_decomm = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.edge == true && attr.in_commission == false }
  region_to_edge_dc_map_incomm = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.edge == true && attr.in_commission == true }
  region_to_edge_dc_map_decomm = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.edge == true && attr.in_commission == false }

  all_dc_to_region_map        = { for dc, attr in local.datacenters : dc => attr.aws_region }
  region_to_all_dc_map        = { for dc, attr in local.datacenters : attr.aws_region => dc... }
  all_dc_to_region_map_incomm = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.in_commission == true }
  all_dc_to_region_map_decomm = { for dc, attr in local.datacenters : dc => attr.aws_region if attr.in_commission == false }
  region_to_all_dc_map_incomm = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.in_commission == true }
  region_to_all_dc_map_decomm = { for dc, attr in local.datacenters : attr.aws_region => dc... if attr.in_commission == false }
}

output "origin_datacenters" {
  description = "A list of origin datacenters."
  value       = local.origin_dcs
}

output "origin_datacenters_in_commission" {
  description = "A list of origin datacenters that are physically in commission."
  value       = local.origin_dcs_incomm
}

output "origin_datacenters_decommissioned" {
  description = "A list of origin datacenters that are physically decommissioned."
  value       = local.origin_dcs_decomm
}

output "edge_datacenters" {
  description = "A list of edge datacenters aka POPs."
  value       = local.edge_dcs
}

output "edge_datacenters_in_commission" {
  description = "A list of edge datacenters that are physically in commission."
  value       = local.edge_dcs_incomm
}

output "edge_datacenters_decommissioned" {
  description = "A list of edge datacenters that are physically decommissioned."
  value       = local.edge_dcs_decomm
}

output "all_datacenters" {
  description = "A list of all datacenters (including origins and edges)."
  value       = sort(keys(local.datacenters))
}

output "global_map_by_datacenters" {
  description = "A map of datacenters grouped by regions, by edge/origin/all."
  value = {
    edge   = local.edge_dc_to_region_map
    origin = local.origin_dc_to_region_map
    all    = local.all_dc_to_region_map
  }
}

output "global_map_by_datacenters_in_commission" {
  description = "A map of physically in commission datacenters grouped by regions, by edge/origin/all."
  value = {
    edge   = local.edge_dc_to_region_map_incomm
    origin = local.origin_dc_to_region_map_incomm
    all    = local.all_dc_to_region_map_incomm
  }
}

output "global_map_by_datacenters_decommissioned" {
  description = "A map of physically decommissioned datacenters grouped by regions, by edge/origin/all."
  value = {
    edge   = local.edge_dc_to_region_map_decomm
    origin = local.origin_dc_to_region_map_decomm
    all    = local.all_dc_to_region_map_decomm
  }
}

output "global_map_by_regions" {
  description = "A map of datacenters grouped by AWS regions, by edge/origin/all."
  value = {
    edge   = local.region_to_edge_dc_map
    origin = local.region_to_origin_dc_map
    all    = local.region_to_all_dc_map
  }
}

output "global_map_by_regions_in_commission" {
  description = "A map of physically in commission datacenters grouped by AWS regions, by edge/origin/all."
  value = {
    edge   = local.region_to_edge_dc_map_incomm
    origin = local.region_to_origin_dc_map_incomm
    all    = local.region_to_all_dc_map_incomm
  }
}

output "global_map_by_regions_decommissioned" {
  description = "A map of physically decommissioned datacenters grouped by AWS regions, by edge/origin/all."
  value = {
    edge   = local.region_to_edge_dc_map_decomm
    origin = local.region_to_origin_dc_map_decomm
    all    = local.region_to_all_dc_map_decomm
  }
}

output "datacenters_map" {
  description = "The raw map of datacenters and its attributes. Useful if the existing outputs do not provide the necessary transformation required."
  value       = local.datacenters
}
