import logging
import boto3
import json
import sys
import os

from botocore.exceptions import ClientError
from distutils import util


# Logging (replace default)
logger = logging.getLogger()
if logger.handlers:
    for handler in logger.handlers:
        logger.removeHandler(handler)
logging.basicConfig(stream=sys.stdout, level=logging.INFO)

# Default ttl of DNS records in seconds
DEFAULT_TTL = 300


class DnsUpdater(object):
    '''Manage handling of DNS'''

    def __init__(self,
                 region,
                 domains,
                 r53_assume,
                 allowed_roles,
                 ttl=DEFAULT_TTL,
                 pop_subdomain=True):

        self.region = region
        self.domains = domains
        self.r53_assume = r53_assume
        self.allowed_roles = allowed_roles
        self.pop_subdomain = pop_subdomain
        self.ttl = ttl

        # Instantiate clients
        self.sts = boto3.client('sts', region_name=region, endpoint_url="https://sts.{}.amazonaws.com".format(region))

    # Upsert the route 53 IVS name.
    # Return True if update was successful. Return False otherwise
    # NOTE: Any errors with this function will result in the lambda exiting, forcing messages to DLQ
    def update_dns(self, contents, source):
        # Connectivity and creds
        try:
            # Assume that role
            sts_creds = self.sts.assume_role(
                RoleArn=self.r53_assume,
                RoleSessionName="R53CrudSession"
            )
            r53 = boto3.client(
                'route53',
                aws_access_key_id=sts_creds.get('Credentials', {}).get('AccessKeyId'),
                aws_secret_access_key=sts_creds.get('Credentials', {}).get('SecretAccessKey'),
                aws_session_token=sts_creds.get('Credentials', {}).get('SessionToken'),
            )
        except ClientError as e:
            logger.error(" Failed Route53 connection steps: %s", e)
            return False

        for domain in self.domains:
            # form subdomain split by pop if required
            subdomain = f"{contents.get('pop')}.{domain}" if self.pop_subdomain else domain

            # get R53 Zone ID
            try:
                # thanks christi LOL
                hosted_zone = r53.list_hosted_zones_by_name(DNSName=subdomain, MaxItems="1").get('HostedZones')[0]

                # Route53 returns partial match if DNSName does not match completely
                if hosted_zone.get('Name') != f"{subdomain}.":
                    raise ValueError("Hosted zone found does not match", hosted_zone.get('Name'))

                hosted_zone_id = hosted_zone.get('Id').split('/')[2]
            except (ClientError, IndexError, ValueError) as e:
                logger.error(" Failed to find zone ID for %s: %s", subdomain, e)
                return False  # weird error so don't even bother retrying. Directly to normal DLQ

            # upsert record
            ivs_fqdn = "{}.{}".format(contents.get('hostname').split('.')[0], subdomain)
            try:
                r53.change_resource_record_sets(
                    HostedZoneId=hosted_zone_id,
                    ChangeBatch={
                        "Comment": "CRUD Lambda update beep boop",
                        "Changes": [{
                            "Action": "UPSERT",
                            "ResourceRecordSet": {
                                "Name": ivs_fqdn,
                                "Type": "A",
                                "TTL": self.ttl,
                                "ResourceRecords": [{"Value": contents.get('new_ip_address')}]
                            }
                        }]
                    }
                )
            except ClientError as e:
                logger.error(" Failed to upsert %s: %s", ivs_fqdn, e)
                return False

            # Let's log successes to make searching easier
            logger.info(" %s (subdomain %s) upserted successfully with IP %s",
                        ivs_fqdn, subdomain, contents.get('new_ip_address'))

        return True

    # Decide how a message is handled depending on its contents
    # Return True if message was processed succesfully. Return False otherwise
    def handle_message(self, record, source):
        # load in the message
        try:
            raw_message = json.loads(record)
            # Legacy, kept this ternary around as it simplifies loading raw (for testing) vs (SNS -> SQS) payloads
            # Dumb ternary because the direct SQS SendMessage format is simpler than the SNS -> SQS message format
            # which wraps the message json into the 'Message' key
            contents = json.loads(raw_message.get('Message')) if 'Message' in raw_message else raw_message
        except json.JSONDecodeError as e:
            logger.error(" Failed to load message json: %s", e)
            return False

        if contents.get('twitch_role') in self.allowed_roles:
            # Allowed. Proceed to update DNS
            return self.update_dns(contents, source)
        else:
            # noop for non-allowed roles
            logger.info(" Role %s not allowed. Skipping", contents.get('twitch_role'))
            return True


def lambda_handler(event, context):
    # Through event source mappings, multiple messages may be processed

    # Lambda Environment
    region = os.environ.get('AWS_REGION')
    domains = os.environ.get('TARGET_DNS_DOMAINS').split(',')
    r53_assume = os.environ.get('R53_ASSUME_ROLE')
    allowed_roles = os.environ.get('ALLOW_LIST_MACHINE_CLASSES').split('|')
    pop_subdomain = bool(util.strtobool(os.environ.get('POP_SUBDOMAIN', 'false')))
    ttl = int(os.environ.get('TTL', DEFAULT_TTL))

    dns_updater = DnsUpdater(region=region, domains=domains, r53_assume=r53_assume,
                             allowed_roles=allowed_roles, pop_subdomain=pop_subdomain,
                             ttl=ttl)

    for record in event.get('Records'):
        if not dns_updater.handle_message(record.get('body'), record.get('eventSourceARN')):
            # Even if 1 message in the batch fails, fail completely
            # 'All messages in a failed batch return to the queue, so your function code must be able to process the same message multiple times without side effects' # noqa: E501
            # https://docs.aws.amazon.com/lambda/latest/dg/with-sqs.html
            sys.exit(1)

    logger.info(" Processed %s messages", len(event.get('Records')))
