# import boto3
import os
import pytest
import logging
import json

from .. import dns_updater
from moto import mock_sts

AWS_REGION = "us-west-2"

#################
# Payloads
#################

HOST = "video-edge-c2a9ac"
POP = "rio01"
FQDN = "{}.{}.justin.tv".format(HOST, POP)
TWITCH_ROLE = "video-edge"
IP = "52.223.208.18"
TARGET_DOMAINS = "warp.live-video.net,abs.hls.ttvnw.net"
R53_ASSUME = "arn:aws:iam::393787499934:role/hls-crud-lambda"

ALLOW_LIST_MACHINE_CLASSES = "video-edge|video-edge-staging|video-warp"


# Sample payload from DHCP hook published to SNS
def dhcp_payload(host=HOST, twitch_role=TWITCH_ROLE, new_ip=IP, pop=POP):
    return json.dumps({
        "source": "dhclient",
        "reason": "RENEW",
        "hostname": "{}.{}.justin.tv".format(host, pop),
        "timestamp": 1625435276,
        "pop": pop,
        "twitch_role": twitch_role,
        "twitch_environment": "production",
        "old_ip_address": "1.2.3.4",  # can be anything
        "new_ip_address": new_ip,
    })


# Sample event from from dhcp hook SNS -> SQS
def render_event(payload=dhcp_payload()):
    return {
        "Records": [{
            "body": json.dumps({
                "Message": payload,
            }),
            "eventSourceARN": "arn:aws:sqs:us-west-2:123456789012:MyQueue",
        }]
    }


#########################
# Basic components setup
#########################


@pytest.fixture
def test_hosted_zone(r53_client):
    for target_domain in TARGET_DOMAINS.split(","):
        r53_client.create_hosted_zone(Name="{}.{}".format(POP, target_domain), CallerReference=target_domain)


#########################
# Tests
#########################


@pytest.fixture(autouse=True)
def setup(test_hosted_zone, caplog):
    """
    Setups environment variables for use for all tests
    """
    os.environ["AWS_REGION"] = AWS_REGION

    os.environ["R53_ASSUME_ROLE"] = R53_ASSUME
    os.environ["TARGET_DNS_DOMAINS"] = TARGET_DOMAINS
    os.environ["ALLOW_LIST_MACHINE_CLASSES"] = ALLOW_LIST_MACHINE_CLASSES
    os.environ["POP_SUBDOMAIN"] = "true"

    caplog.set_level(logging.INFO)


@mock_sts
def test_handler_ideal(caplog):
    """
    Simulate a trigger from SQS with a valid machine
    """

    status = dns_updater.lambda_handler(event=render_event(), context={})
    for target_domain in TARGET_DOMAINS.split(','):
        assert ' {0}.{1}.{2} (subdomain {1}.{2}) upserted successfully with IP {3}'.format(HOST, POP, target_domain, IP) in caplog.messages  # noqa: E501
        assert status is None


@mock_sts
def test_handler_invalid_zone():
    """
    What happens if route53 zone does not exist
    """

    with pytest.raises(SystemExit) as pytest_wrapped_e:
        dns_updater.lambda_handler(event=render_event(payload=dhcp_payload(pop="DNE01")), context={})

    assert pytest_wrapped_e.type == SystemExit
    assert pytest_wrapped_e.value.code == 1


@mock_sts
def test_handler_not_pop_subdomain(caplog, r53_client):
    """
    Test regular zone (not by pop). E.g. <host>.foo.bar.baz
    """

    os.environ["POP_SUBDOMAIN"] = "false"

    for target_domain in TARGET_DOMAINS.split(','):
        r53_client.create_hosted_zone(Name=target_domain, CallerReference=target_domain)

    status = dns_updater.lambda_handler(event=render_event(), context={})
    assert status is None

    for target_domain in TARGET_DOMAINS.split(','):
        assert f" {HOST}.{target_domain} (subdomain {target_domain}) upserted successfully with IP {IP}" in caplog.messages  # noqa: E501


@mock_sts
def test_subdomain_ttl(r53_client):

    dns_updater.lambda_handler(event=render_event(), context={})

    hosted_zones = r53_client.list_hosted_zones()
    hosted_zone_id = hosted_zones.get('HostedZones')[0].get('Id').split('/')[2]
    records = r53_client.list_resource_record_sets(HostedZoneId=hosted_zone_id, MaxItems="1")

    # Default TTL
    assert 300 == records["ResourceRecordSets"][0]["TTL"]

    # Override TTL
    os.environ["TTL"] = "3600"
    dns_updater.lambda_handler(event=render_event(), context={})
    records = r53_client.list_resource_record_sets(HostedZoneId=hosted_zone_id, MaxItems="1")

    assert 3600 == records["ResourceRecordSets"][0]["TTL"]


class TestDnsUpdater(object):
    def setup_class(self):
        self.dns_updater = dns_updater.DnsUpdater(region=AWS_REGION,
                                                  domains=TARGET_DOMAINS,
                                                  r53_assume=R53_ASSUME,
                                                  allowed_roles=ALLOW_LIST_MACHINE_CLASSES.split("|"))

    def test_handle_message_invalid_machine_class(self, caplog):
        record = dhcp_payload(twitch_role="NOPE_NOPE")
        assert self.dns_updater.handle_message(record=record, source="test") is True
        assert " Role NOPE_NOPE not allowed. Skipping" in caplog.messages

    def test_handle_message_missing_machine_class(self, caplog):
        record = json.dumps({})
        assert self.dns_updater.handle_message(record=record, source="test") is True
        assert " Role None not allowed. Skipping" in caplog.messages
