# the queue that will receive cross account SNS notifications
resource "aws_sqs_queue" "service_queue" {
  name                       = "${var.service_id}-queue"
  visibility_timeout_seconds = 900 # 15 mins
  delay_seconds              = 0   # no delay
  tags                       = local.common_tags
  message_retention_seconds  = 86400 # one day
  redrive_policy             = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.service_deadletter.arn}\",\"maxReceiveCount\":3}"
}

# the deadletter queue is where failed to process messages (processed unsuccessfully > 3 times)
# get sent to for debugging
resource "aws_sqs_queue" "service_deadletter" {
  name                      = "${var.service_id}-deadletter"
  message_retention_seconds = 604800 # seven days
  tags                      = local.common_tags
}

# Allow sns topic to send messages
resource "aws_sqs_queue_policy" "allow_sns_to_send" {
  queue_url = aws_sqs_queue.service_queue.id
  policy    = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "SQSPolicy",
  "Statement": [
    {
      "Sid": "AllowSNSToSend",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "sqs:SendMessage",
      "Resource": "${aws_sqs_queue.service_queue.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${var.input_sns_arn}"
        }
      }
    },
    {
      "Sid": "AllowLambdaActions",
      "Effect": "Allow",
      "Principal": "*",
      "Action": [
        "sqs:ReceiveMessage",
        "sqs:DeleteMessage",
        "sqs:GetQueueAttributes"
      ],
      "Resource": "${aws_sqs_queue.service_queue.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${aws_lambda_function.service_dns_lambda.arn}"
        }
      }
    }
  ]
}
POLICY
}

# Subscribe to the input SNS topic (you must configure the input SNS topic to allow the account you're running this tf with to subscribe).
# See step 5 https://docs.aws.amazon.com/sns/latest/dg/sns-send-message-to-sqs-cross-account.html#SendMessageToSQS.cross.account.queueowner
resource "aws_sns_topic_subscription" "user_updates_sqs_target" {
  topic_arn = var.input_sns_arn
  protocol  = "sqs"
  endpoint  = aws_sqs_queue.service_queue.arn
}
