# Role used for account-level ECS services.
resource "aws_iam_role" "ecs_task_role" {
  name_prefix = "ecs_task_role_${local.account}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "cloudwatch_metrics_policy" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = "arn:aws:iam::aws:policy/CloudWatchAgentServerPolicy"
}

# SNS topic

resource "aws_sns_topic" "graphite_sns_topic" {
  name = "${var.team_name}-${var.env}-${var.region}-graphite-stack-alarm-topic"
}


resource "aws_sns_topic_subscription" "pd_subscription" {
  topic_arn              = aws_sns_topic.graphite_sns_topic.arn
  protocol               = "https"
  endpoint               = var.pagerduty_url
  endpoint_auto_confirms = true
}

module "core_account" {
  source = "git::git+ssh://git@git.xarth.tv/video-coreservices/tf-mod-voncount-core-account?ref=v1.1.0"

  environment = local.account
  team        = var.team_name
  xray_role   = "*"
  # write to all the xray's
}

module "go_carbon_a" {
  source = "./modules/go_carbon"

  cluster_size = {
    min     = var.go_carbon_node_count
    max     = var.go_carbon_node_count
    desired = var.go_carbon_node_count
  }
  go_carbon_instance_type = var.go_carbon_instance_type
  go_carbon_storage_size  = var.go_carbon_storage_size
  ebs_iops                = var.ebs_iops

  vpc_id          = var.vpc_id
  subnets         = [var.graphite_subnets[0]]
  security_groups = var.security_groups

  environment  = local.account
  cluster_name = "go_carbon_a"
  image_url    = "${local.account_id}.dkr.ecr.${var.region}.amazonaws.com/go_carbon:${var.go_carbon_image_tag}"
  region       = var.region
  account_id   = local.account_id

  owner_email = var.owner
  team_name   = var.team_name

  service_iam_role              = module.core_account.service_iam_role
  container_instance_profile_id = module.core_account.container_instance_profile_id
  ecs_task_role_arn             = aws_iam_role.ecs_task_role.arn
  alarm_sns_topics              = [aws_sns_topic.graphite_sns_topic.arn]
}

module "go_carbon_b" {
  source = "./modules/go_carbon"

  cluster_size = {
    min     = var.go_carbon_node_count
    max     = var.go_carbon_node_count
    desired = var.go_carbon_node_count
  }
  go_carbon_instance_type = var.go_carbon_instance_type
  go_carbon_storage_size  = var.go_carbon_storage_size
  ebs_iops                = var.ebs_iops

  vpc_id          = var.vpc_id
  subnets         = [var.graphite_subnets[1]]
  security_groups = var.security_groups

  environment  = local.account
  cluster_name = "go_carbon_b"
  image_url    = "${local.account_id}.dkr.ecr.${var.region}.amazonaws.com/go_carbon:${var.go_carbon_image_tag}"
  region       = var.region
  account_id   = local.account_id

  owner_email = var.owner
  team_name   = var.team_name

  service_iam_role              = module.core_account.service_iam_role
  container_instance_profile_id = module.core_account.container_instance_profile_id
  ecs_task_role_arn             = aws_iam_role.ecs_task_role.arn
  alarm_sns_topics              = [aws_sns_topic.graphite_sns_topic.arn]
}

module "graphite-web" {
  source = "./modules/graphite-web"

  providers = {
    template = template
  }

  cluster_size = {
    min     = var.graphite_web_node_count
    max     = var.graphite_web_node_count
    desired = var.graphite_web_node_count
  }
  graphite_web_instance_type = var.graphite_web_instance_type

  vpc_id          = var.vpc_id
  subnets         = var.graphite_subnets
  security_groups = var.security_groups

  lb_name      = "graphite-web"
  environment  = local.account
  cluster_name = "graphite-web"
  image_url    = "${local.account_id}.dkr.ecr.${var.region}.amazonaws.com/graphite_web:${var.graphite_web_image_tag}"
  region       = var.region
  account_id   = local.account_id

  owner_email = var.owner
  team_name   = var.team_name

  service_iam_role              = module.core_account.service_iam_role
  container_instance_profile_id = module.core_account.container_instance_profile_id
  ecs_task_role_arn             = aws_iam_role.ecs_task_role.arn
  alarm_sns_topics              = [aws_sns_topic.graphite_sns_topic.arn]

  depends_on = [
    module.go_carbon_a,
    module.go_carbon_b,
  ]
}

module "carbon-c-relay" {
  source = "./modules/carbon-c-relay"

  providers = {
    template = template
  }


  cluster_size = {
    min     = var.carbon_c_relay_node_count
    max     = var.carbon_c_relay_node_count
    desired = var.carbon_c_relay_node_count
  }
  carbon_c_relay_instance_type = var.carbon_c_relay_instance_type

  vpc_id          = var.vpc_id
  subnets         = var.graphite_subnets
  security_groups = var.security_groups

  nlb_name     = "carbon-c-relay"
  environment  = local.account
  cluster_name = "carbon-c-relay"
  image_url    = "${local.account_id}.dkr.ecr.${var.region}.amazonaws.com/carbon_c_relay:${var.carbon_c_relay_image_tag}"
  region       = var.region
  account_id   = local.account_id

  owner_email = var.owner
  team_name   = var.team_name

  service_iam_role              = module.core_account.service_iam_role
  container_instance_profile_id = module.core_account.container_instance_profile_id
  ecs_task_role_arn             = aws_iam_role.ecs_task_role.arn
  alarm_sns_topics              = [aws_sns_topic.graphite_sns_topic.arn]

  depends_on = [
    module.go_carbon_a,
    module.go_carbon_b,
  ]
}
