locals {
  metric_namespace = "CarbonRelayNG"
}

resource "aws_cloudwatch_log_metric_filter" "invalid_credentials" {
  name           = "invalid_credential_error_count"
  pattern        = "InvalidCredentials"
  log_group_name = aws_cloudwatch_log_group.app_logs.name

  metric_transformation {
    name      = "InvalidCredentials"
    namespace = local.metric_namespace
    value     = "1"
  }
}

resource "aws_cloudwatch_log_metric_filter" "submit_data_failure" {
  name           = "submit_data_failure"
  pattern        = "GrafanaNet failed to submit data"
  log_group_name = aws_cloudwatch_log_group.app_logs.name

  metric_transformation {
    name      = "SubmitDataFailure"
    namespace = local.metric_namespace
    value     = "1"
  }
}

resource "aws_cloudwatch_metric_alarm" "cpu_utilization" {
  alarm_name          = "carbon_c_relay_cpu_utilization"
  alarm_description   = "Alarm when carbon-c-relay cpu utilization is > 80% for 5 minute. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-carbon_c_relay_cpu_utilization"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "CPUUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  treat_missing_data = "missing"
  alarm_actions      = var.alarm_sns_topics
  ok_actions         = var.alarm_sns_topics
}

resource "aws_cloudwatch_metric_alarm" "memory_utilization" {
  alarm_name          = "carbon_c_relay_memory_utilization"
  alarm_description   = "Alarm when carbon-c-relay memory utilization is > 80% for 5 minute. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-carbon_c_relay_memory_utilization"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "MemoryUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  treat_missing_data = "missing"
  alarm_actions      = var.alarm_sns_topics
  ok_actions         = var.alarm_sns_topics
}

resource "aws_cloudwatch_metric_alarm" "running_tasks" {
  alarm_name          = "carbon_c_relay_running_tasks"
  alarm_description   = "Alarm when carbon-c-relay has fewer than expected running tasks. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-carbon_c_relay_running_tasks"
  comparison_operator = "LessThanThreshold"
  // Count the number of tasks that are reporting CPU utilization.
  metric_name = "CPUUtilization"
  statistic   = "SampleCount"
  namespace   = "AWS/ECS"
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = local.tasks_per_instance * var.cluster_size.desired
  treat_missing_data = "missing"
  alarm_actions      = var.alarm_sns_topics
  ok_actions         = var.alarm_sns_topics
}
