locals {
  metric_namespace = "CarbonRelayNG"
}


resource "aws_cloudwatch_metric_alarm" "cpu_utilization" {
  alarm_name          = "${var.cluster_name}_cpu_utilization"
  alarm_description   = "Alarm when go-carbon cpu utilization is > 80% for 5 minute. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-go_carbon_cpu_utilization"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "CPUUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  alarm_actions       = var.alarm_sns_topics
  ok_actions          = var.alarm_sns_topics
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  treat_missing_data = "missing"
}

resource "aws_cloudwatch_metric_alarm" "memory_utilization" {
  alarm_name          = "${var.cluster_name}_memory_utilization"
  alarm_description   = "Alarm when go-carbon memory utilization is > 80% for 5 minute. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-go_carbon_memory_utilization"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  metric_name         = "MemoryUtilization"
  statistic           = "Average"
  namespace           = "AWS/ECS"
  alarm_actions       = var.alarm_sns_topics
  ok_actions          = var.alarm_sns_topics
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = "80"
  treat_missing_data = "missing"
}

resource "aws_cloudwatch_metric_alarm" "running_tasks" {
  alarm_name          = "${var.cluster_name}_running_tasks"
  alarm_description   = "Alarm when go-carbon has fewer than expected running tasks. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-go_carbon_running_tasks"
  comparison_operator = "LessThanThreshold"
  // Count the number of tasks that are reporting CPU utilization.
  metric_name   = "CPUUtilization"
  statistic     = "SampleCount"
  namespace     = "AWS/ECS"
  alarm_actions = var.alarm_sns_topics
  ok_actions    = var.alarm_sns_topics
  dimensions = {
    ClusterName = var.cluster_name
    ServiceName = module.ecs.ecs_service_name
  }

  evaluation_periods = "5"
  period             = "60"
  threshold          = local.tasks_per_instance * var.cluster_size.desired
  treat_missing_data = "missing"
}

resource "aws_cloudwatch_metric_alarm" "asg_max_disk_space" {
  alarm_name          = "${var.cluster_name}_disk_space"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = "5"
  datapoints_to_alarm = "5"
  threshold           = "90"
  alarm_description   = "An instance in the ASG has exceeded 90% disk space. https://wiki.xarth.tv/display/VID/SHG+Metrics+and+Alarming#SHGMetricsandAlarming-go_carbon_disk_space"
  alarm_actions       = var.alarm_sns_topics
  ok_actions          = var.alarm_sns_topics
  treat_missing_data  = "missing"

  metric_query {
    id          = "m1"
    return_data = "true"
    metric {
      metric_name = "disk_used_percent"
      namespace   = "GoCarbon"
      period      = "60"
      stat        = "Maximum"

      dimensions = {
        AutoScalingGroupName = "${var.team_name}-graphite-${var.cluster_name}"
      }
    }
  }

}
