# TODO move this module to the shared terraform repo. This is blocked on the
# "network_lb" module being available in that repo
locals {
  app_name           = "go_carbon"
  app_port           = 2003 # Traffic to this port is expect to conform to the carbon protocol
  lb_port            = 2003 # Use the carbon port at the load balancer to match expectations and convention
  memory             = 6000
  ulimit             = 200000
  instance_type      = var.go_carbon_instance_type
  tasks_per_instance = 1
}

resource "aws_cloudwatch_log_group" "app_logs" {
  name              = "${var.cluster_name}-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app = local.app_name
    env = var.environment
  }
}

data "template_cloudinit_config" "user_data" {
  gzip          = false
  base64_encode = false

  part {
    filename     = "init.cfg"
    content_type = "text/cloud-config"

    content = templatefile("${path.module}/templates/cloudinit.yaml.tmpl", {
      service    = "graphite"
      dns_region = var.region
    })
  }

  part {
    content_type = "text/x-shellscript"

    content = templatefile("${path.module}/templates/startup.sh.tmpl", {
      cluster_name = var.cluster_name
      owner        = "vidcs"
      service      = "graphite"
      dns_region   = var.region
    })
  }
}

module "ecs" {
  source = "../ecs"

  app_name     = var.cluster_name
  environment  = var.environment
  cluster_name = var.cluster_name
  user_data    = data.template_cloudinit_config.user_data.rendered

  task_network_mode = "host"
  ec2_instance_type = local.instance_type
  app_count         = local.tasks_per_instance * var.cluster_size.desired
  cluster_size      = var.cluster_size

  security_groups = join(",", var.security_groups)
  subnets         = var.subnets

  ecs_host_iam_id = var.container_instance_profile_id

  app_port              = local.app_port
  container_definitions = <<DEFINITION
[
  {
    "name": "${var.cluster_name}",
    "image": "${var.image_url}",
    "cpu": 0,
    "memory": ${local.memory},
    "essential": true,
    "portMappings": [
      {
        "protocol": "tcp",
        "containerPort": ${local.app_port},
        "hostPort": ${local.app_port}
      }
    ],
    "environment": [
      {
        "name": "GRAPHITE_HOST_PORT",
        "value": "localhost:2003"
      }
    ],
    "logConfiguration": {
      "logDriver": "awslogs",
      "secretOptions": null,
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.app_logs.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${local.app_name}"
      }
    },
    "volumesFrom": [],
    "mountPoints": [
      {
        "sourceVolume": "metrics",
        "containerPath": "/metrics",
        "readOnly": false
      }
    ],
    "ulimits": [
      {
        "name": "nofile",
        "softLimit": ${local.ulimit},
        "hardLimit": ${local.ulimit}
      }
    ]
  }
]
DEFINITION

  use_dnsmasq = false

  ecs_task_role = var.ecs_task_role_arn

  owner_email      = var.owner_email
  service_iam_role = var.service_iam_role
  team_name        = var.team_name
  host_volume = [
    {
      name      = "metrics"
      host_path = "/storage/metrics"
    }
  ]
}

data "aws_subnet" "storage_subnet" {
  id = var.subnets[0]
}

resource "aws_ebs_volume" "graphite_data" {
  availability_zone = data.aws_subnet.storage_subnet.availability_zone
  size              = var.go_carbon_storage_size
  count             = var.cluster_size.desired
  type              = "gp3"
  iops              = var.ebs_iops
  tags = {
    name  = "graphite_data",
    index = count.index
  }
}

resource "aws_network_interface" "eni" {
  count           = var.cluster_size.desired
  subnet_id       = var.subnets[0]
  security_groups = var.security_groups

  tags = {
    name  = "graphite"
    index = count.index
  }
}
