# TODO move this module to the shared terraform repo. This is blocked on the
# "network_lb" module being available in that repo
locals {
  app_name           = "graphite-web"
  app_port           = 8000
  lb_port            = 80
  memory             = 4096
  ulimit             = 100000
  instance_type      = var.graphite_web_instance_type
  tasks_per_instance = 1
}

resource "aws_cloudwatch_log_group" "app_logs" {
  name              = "${var.cluster_name}-logs"
  retention_in_days = var.log_retention_days

  tags = {
    app = local.app_name
    env = var.environment
  }
}

data "template_cloudinit_config" "user_data" {
  gzip          = false
  base64_encode = false

  part {
    filename     = "init.cfg"
    content_type = "text/cloud-config"

    content = templatefile("${path.module}/templates/cloudinit.yaml.tmpl", {
      service    = "graphite"
      dns_region = var.region
    })
  }

  part {
    content_type = "text/x-shellscript"

    content = templatefile("${path.module}/templates/startup.sh.tmpl", {
      cluster_name = var.cluster_name
      owner        = "vidcs"
      service      = "graphite"
      dns_region   = var.region
    })
  }
}

module "ecs" {
  source = "../ecs"

  app_name     = local.app_name
  environment  = var.environment
  cluster_name = var.cluster_name
  user_data    = data.template_cloudinit_config.user_data.rendered

  task_network_mode = "host"
  ec2_instance_type = local.instance_type
  app_count         = local.tasks_per_instance * var.cluster_size.desired
  cluster_size      = var.cluster_size

  security_groups = join(",", var.security_groups)
  subnets         = var.subnets

  ecs_host_iam_id = var.container_instance_profile_id

  app_port              = local.app_port
  container_definitions = <<DEFINITION
[
  {
    "name": "${var.cluster_name}",
    "image": "${var.image_url}",
    "cpu": 0,
    "memory": ${local.memory},
    "essential": true,
    "portMappings": [
      {
        "protocol": "tcp",
        "containerPort": ${local.app_port},
        "hostPort": ${local.app_port}
      }
    ],
    "environment": [],
    "volumesFrom": [],
    "logConfiguration": {
      "logDriver": "awslogs",
      "secretOptions": null,
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.app_logs.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${local.app_name}"
      }
    },
    "ulimits": [
      {
        "name": "nofile",
        "softLimit": ${local.ulimit},
        "hardLimit": ${local.ulimit}
      }
    ]
  }
]
DEFINITION

  use_dnsmasq = false

  ecs_task_role = var.ecs_task_role_arn

  target_group_ids = [aws_lb_target_group.main.id]

  owner_email      = var.owner_email
  service_iam_role = var.service_iam_role
  team_name        = var.team_name
}

resource "aws_lb" "main" {
  name               = var.lb_name
  internal           = true
  load_balancer_type = "application"
  subnets            = var.subnets
  security_groups    = var.security_groups
}

resource "aws_lb_target_group" "main" {
  deregistration_delay = 5 # seconds
  slow_start           = 0
  name                 = aws_lb.main.name
  protocol             = "HTTP"
  vpc_id               = var.vpc_id

  target_type = "instance"

  # The port is required, but overriden by ECS. Using a placeholder here.
  port = local.app_port

  health_check {
    protocol            = "HTTP"
    healthy_threshold   = 3  # checks resulting in success
    unhealthy_threshold = 3  # checks resulting in failure
    interval            = 10 # seconds. Only 10 and 30 are supported
    # custom timeouts and custom status codes are not support. healthy and unhealthy must be the same
  }
}

# Redirect all traffic from the LB to the target group
resource "aws_lb_listener" "main" {
  load_balancer_arn = aws_lb.main.id
  port              = local.lb_port
  protocol          = "HTTP"

  default_action {
    target_group_arn = aws_lb_target_group.main.id
    type             = "forward"
  }
}
