data "aws_partition" "current" {}

resource "aws_s3_bucket" "root_crl" {
  bucket = "${var.service-name}-root-crl"
  website {
    index_document = "index.html"
  }
  tags = local.tags
}

data "aws_iam_policy_document" "root_crl" {
  # TODO: allow read access from twitch subnets?
  statement {
    actions = [
      "s3:GetBucketAcl",
      "s3:GetBucketLocation",
      "s3:PutObject",
      "s3:PutObjectAcl",
    ]

    resources = [
      aws_s3_bucket.root_crl.arn,
      "${aws_s3_bucket.root_crl.arn}/*",
    ]

    principals {
      identifiers = ["acm-pca.amazonaws.com"]
      type        = "Service"
    }
  }
}

resource "aws_s3_bucket_policy" "root_crl" {
  bucket = aws_s3_bucket.root_crl.id
  policy = data.aws_iam_policy_document.root_crl.json
}

resource "aws_acmpca_certificate_authority" "root" {
  type = "ROOT"
  tags = local.tags
  certificate_authority_configuration {
    key_algorithm     = var.ca-key-algo
    signing_algorithm = var.signing-algo

    subject {
      common_name = "${var.service-name} Root CA G${var.root-ca-generation}"
    }
  }

  revocation_configuration {
    crl_configuration {
      custom_cname       = "${var.service-name}-root-crl.s3-website-${var.region}.amazonaws.com"
      enabled            = true
      expiration_in_days = var.crl-expiration
      s3_bucket_name     = aws_s3_bucket.root_crl.id
    }
  }

  depends_on = [aws_s3_bucket_policy.root_crl]
}

resource "aws_acmpca_certificate_authority_certificate" "root" {
  certificate_authority_arn = aws_acmpca_certificate_authority.root.arn

  certificate       = aws_acmpca_certificate.root.certificate
  certificate_chain = aws_acmpca_certificate.root.certificate_chain
}

resource "aws_acmpca_certificate" "root" {
  certificate_authority_arn   = aws_acmpca_certificate_authority.root.arn
  certificate_signing_request = aws_acmpca_certificate_authority.root.certificate_signing_request
  signing_algorithm           = var.signing-algo

  template_arn = "arn:${data.aws_partition.current.partition}:acm-pca:::template/RootCACertificate/V1"

  validity {
    type  = "YEARS"
    value = var.root-ca-validity
  }
}


resource "aws_s3_bucket" "intermediate_crl" {
  bucket = "${var.service-name}-intermediate-crl"
  website {
    index_document = "index.html"
  }
  tags = local.tags
}

data "aws_iam_policy_document" "intermediate_crl" {
  # TODO: allow read access from twitch subnets?
  statement {
    actions = [
      "s3:GetBucketAcl",
      "s3:GetBucketLocation",
      "s3:PutObject",
      "s3:PutObjectAcl",
    ]

    resources = [
      aws_s3_bucket.intermediate_crl.arn,
      "${aws_s3_bucket.intermediate_crl.arn}/*",
    ]

    principals {
      identifiers = ["acm-pca.amazonaws.com"]
      type        = "Service"
    }
  }
}

resource "aws_s3_bucket_policy" "intermediate_crl" {
  bucket = aws_s3_bucket.intermediate_crl.id
  policy = data.aws_iam_policy_document.intermediate_crl.json
}

resource "aws_acmpca_certificate_authority" "intermediate" {
  type = "SUBORDINATE"
  tags = local.tags
  certificate_authority_configuration {
    key_algorithm     = var.ca-key-algo
    signing_algorithm = var.signing-algo

    subject {
      common_name = "${var.service-name} Intermediate CA G${var.intermediate-ca-generation}"
    }
  }

  revocation_configuration {
    crl_configuration {
      custom_cname       = "${var.service-name}-intermediate-crl.s3-website-${var.region}.amazonaws.com"
      enabled            = true
      expiration_in_days = var.crl-expiration
      s3_bucket_name     = aws_s3_bucket.intermediate_crl.id
    }
  }

  depends_on = [aws_s3_bucket_policy.intermediate_crl]
}

resource "aws_acmpca_certificate_authority_certificate" "intermediate" {
  certificate_authority_arn = aws_acmpca_certificate_authority.intermediate.arn

  certificate       = aws_acmpca_certificate.intermediate.certificate
  certificate_chain = aws_acmpca_certificate.intermediate.certificate_chain
}

resource "aws_acmpca_certificate" "intermediate" {
  certificate_authority_arn   = aws_acmpca_certificate_authority.root.arn
  certificate_signing_request = aws_acmpca_certificate_authority.intermediate.certificate_signing_request
  signing_algorithm           = var.signing-algo

  template_arn = "arn:${data.aws_partition.current.partition}:acm-pca:::template/SubordinateCACertificate_PathLen0/V1"

  validity {
    type  = "YEARS"
    value = var.intermediate-ca-validity
  }
}

