resource "aws_iam_policy" "root-maintenance" {
  description = "Restricted access to root CA maintenance"
  policy      = data.aws_iam_policy_document.root-maintenance.json
}

data "aws_iam_policy_document" "root-maintenance" {
  statement {
    effect = "Allow"
    actions = [
      "acm-pca:IssueCertificate",
    ]

    resources = [
      aws_acmpca_certificate_authority.root.arn,
    ]

    condition {
      test     = "StringLike"
      variable = "acm-pca:TemplateArn"

      values = [
        "arn:aws:acm-pca:::template/*CACertificate*/V*",
      ]
    }
  }

  statement {
    effect = "Deny"
    actions = [
      "acm-pca:IssueCertificate",
    ]

    resources = [
      aws_acmpca_certificate_authority.root.arn,
    ]

    condition {
      test     = "StringNotLike"
      variable = "acm-pca:TemplateArn"

      values = [
        "arn:aws:acm-pca:::template/*CACertificate*/V*",
      ]
    }
  }

  statement {
    effect = "Allow"
    actions = [
      "acm-pca:RevokeCertificate",
      "acm-pca:GetCertificate",
      "acm-pca:ListPermissions"
    ]

    resources = [
      aws_acmpca_certificate_authority.root.arn,
    ]

  }

  statement {
    effect = "Allow"
    actions = [
      "acm-pca:ListCertificateAuthorities"
    ]

    resources = [
      "*"
    ]
  }
}

resource "aws_iam_policy" "intermediate-maintenance" {
  description = "Restricted access to intermediate CA maintenance"
  policy      = data.aws_iam_policy_document.intermediate-maintenance.json
}

data "aws_iam_policy_document" "intermediate-maintenance" {
  statement {
    effect = "Allow"
    actions = [
      "acm-pca:IssueCertificate",
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]

    condition {
      test     = "StringLike"
      variable = "acm-pca:TemplateArn"

      values = [
        "arn:aws:acm-pca:::template/*CACertificate*/V*",
      ]
    }
  }

  statement {
    effect = "Deny"
    actions = [
      "acm-pca:IssueCertificate",
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]

    condition {
      test     = "StringNotLike"
      variable = "acm-pca:TemplateArn"

      values = [
        "arn:aws:acm-pca:::template/*CACertificate*/V*",
      ]
    }
  }

  statement {
    effect = "Allow"
    actions = [
      "acm-pca:RevokeCertificate",
      "acm-pca:GetCertificate",
      "acm-pca:ListPermissions"
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]

  }

  statement {
    effect = "Allow"
    actions = [
      "acm-pca:ListCertificateAuthorities"
    ]

    resources = [
      "*"
    ]
  }
}

resource "aws_iam_policy" "service-leaf-issuance" {
  description = "Restricted access to leaf certficate issuance"
  policy      = data.aws_iam_policy_document.service-leaf-issuance.json
}

data "aws_iam_policy_document" "service-leaf-issuance" {
  statement {
    effect = "Allow"
    actions = [
      "acm-pca:IssueCertificate",
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]

    condition {
      test     = "StringLike"
      variable = "acm-pca:TemplateArn"

      values = [
        "arn:aws:acm-pca:::template/EndEntityCertificate/V*"
      ]
    }
  }

  statement {
    effect = "Deny"
    actions = [
      "acm-pca:IssueCertificate",
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]

    condition {
      test     = "StringNotLike"
      variable = "acm-pca:TemplateArn"

      values = [
        "arn:aws:acm-pca:::template/EndEntityCertificate/V*"
      ]
    }
  }

  statement {
    effect = "Allow"
    actions = [
      "acm-pca:RevokeCertificate",
      "acm-pca:GetCertificate",
      "acm-pca:ListPermissions"
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]

  }

  statement {
    effect = "Allow"
    actions = [
      "acm-pca:ListCertificateAuthorities"
    ]

    resources = [
      "*"
    ]
  }
}

resource "aws_iam_policy" "service-intermediate" {
  description = "Organizational access to intermediate certficates"
  policy      = data.aws_iam_policy_document.service-intermediate.json
}

data "aws_iam_policy_document" "service-intermediate" {
  statement {
    effect = "Allow"
    actions = [
      "acm-pca:DescribeCertificateAuthority",
      "acm-pca:DescribeCertificateAuthorityAuditReport",
      "acm-pca:ListCertificateAuthorities",
      "acm-pca:GetCertificateAuthorityCsr",
      "acm-pca:GetCertificateAuthorityCertificate",
      "acm-pca:GetCertificate",
      "acm-pca:GetPolicy",
      "acm-pca:ListPermissions",
      "acm-pca:ListTags"
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]
  }
}

resource "aws_iam_policy" "service-leaf" {
  description = "Service access to leaf certficates"
  policy      = data.aws_iam_policy_document.service-leaf.json
}

data "aws_iam_policy_document" "service-leaf" {
  statement {
    effect = "Allow"
    actions = [
      "acm-pca:DescribeCertificateAuthority",
      "acm-pca:DescribeCertificateAuthorityAuditReport",
      "acm-pca:ListCertificateAuthorities",
      "acm-pca:GetCertificateAuthorityCsr",
      "acm-pca:GetCertificateAuthorityCertificate",
      "acm-pca:GetCertificate",
      "acm-pca:GetPolicy",
      "acm-pca:ListPermissions",
      "acm-pca:ListTags"
    ]

    resources = [
      aws_acmpca_certificate_authority.intermediate.arn,
    ]
  }

  statement {
    effect = "Allow"
    actions = [
      "secretsmanager:Describe*",
      "secretsmanager:List*",
    ]

    resources = [
      "*"
    ]
  }
  statement {
    effect = "Allow"
    actions = [
      "secretsmanager:GetSecretValue"
    ]

    resources = [
      aws_secretsmanager_secret.leaf_key.id
    ]
  }
}

resource "aws_iam_policy" "organization_principal" {
  description = "Organizational assume-role access destination"
  policy      = data.aws_iam_policy_document.organization_principal.json
}

data "aws_iam_policy_document" "organization_principal" {
  statement {
    effect = "Allow"
    principals {
      identifiers = ["*"]
      type        = "AWS"
    }
    actions = ["sts:AssumeRole"]
    // TODO: way to lock down to specific pattern?
    condition {
      test     = "StringEquals"
      variable = "aws:PrincipalOrgID"
      values   = [var.video_organization_id]
    }
  }
}

resource "aws_iam_policy" "icp-intermediate" {
  description = "Organizational assume-role access source"
  policy      = data.aws_iam_policy_document.icp-intermediate.json
}

data "aws_iam_policy_document" "icp-intermediate" {
  statement {
    effect = "Allow"
    actions = ["sts:AssumeRole"]
    // TODO: way to lock down to specific pattern?
    resources = [ "*" ]
  }
}

resource "aws_iam_policy" "service-leaf-assume" {
  description = "Service leaf assume-role access source"
  policy      = data.aws_iam_policy_document.service-leaf-assume.json
}

data "aws_iam_policy_document" "service-leaf-assume" {
  statement {
    effect = "Allow"
    actions = ["sts:AssumeRole"]
    principals {
      identifiers = [ var.service-acct-icp-leaf-role ]
      type        = "AWS"
    }
  }
}

