# ================
## Codebuild IAM
# ================
resource "aws_iam_role" "service_codebuild_role" {
  name = "${var.service_id}-codebuild-service-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "codebuild.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# This policy allows codebuild to access all its necessary resources
# This includes CW logs permissions, cloudformation/iam permissions so the build can trigger a SAM deploy,
# as well as the S3 bucket and keys where our GHE will upload the code/config to kick off the process and codepipeline metadata is stored
resource "aws_iam_policy" "service_codebuild_policy" {
  name = "codebuild-${var.service_id}-service-policy"
  policy = templatefile("${path.module}/files/codebuild_service_policy.json.tpl",
    {
      region                          = var.region,
      caller_account_id               = data.aws_caller_identity.caller.account_id,
      subnets                         = var.subnet_ids == null ? "[]" : "[${join(", ", formatlist("\"arn:aws:ec2:${var.region}:${data.aws_caller_identity.caller.account_id}:subnet/%s\"", var.subnet_ids))}]",
      codesource_bucket_id            = var.codesource_bucket_id,
      service_release_bucket_arn      = aws_s3_bucket.service_release_bucket.arn,
      service_codepipeline_bucket_arn = aws_s3_bucket.service_codepipeline_bucket.arn
    }
  )
}


resource "aws_iam_role_policy_attachment" "service_codebuild_policy_attachment" {
  role       = aws_iam_role.service_codebuild_role.name
  policy_arn = aws_iam_policy.service_codebuild_policy.id
}

# ===================
## Codedeploy IAM
# ===================

resource "aws_iam_role" "service_codedeploy_role" {
  name = "${var.service_id}-codedeploy-service-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
      {
          "Sid": "",
          "Effect": "Allow",
          "Principal": {
              "Service": [
                  "codedeploy.amazonaws.com"
              ]
          },
          "Action": "sts:AssumeRole"
      }
  ]
}
EOF
  tags               = local.common_tags
}

# Use default codedeploy policy
resource "aws_iam_role_policy_attachment" "service_codedeploy_role_policy_attach" {
  role       = aws_iam_role.service_codedeploy_role.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSCodeDeployRoleForLambda"
}

# Extra Codedeploy Policy for S3 bucket access, codedeploy access, and lambda invocations.
resource "aws_iam_policy" "service_codedeploy_allow" {
  name = "codedeploy-${var.service_id}-allow"
  policy = templatefile("${path.module}/files/codedeploy_service_policy.json.tpl",
    {
      codesource_bucket_id            = var.codesource_bucket_id,
      service_release_bucket_arn      = aws_s3_bucket.service_release_bucket.arn,
      service_codepipeline_bucket_arn = aws_s3_bucket.service_codepipeline_bucket.arn
    }
  )
}

# Use default codedeploy policy
resource "aws_iam_role_policy_attachment" "service_codedeploy_role_extra_policy_attach" {
  role       = aws_iam_role.service_codedeploy_role.name
  policy_arn = aws_iam_policy.service_codedeploy_allow.arn
}

# ===================
## Codepipeline IAM
# ===================

# Simple codepipeline role
resource "aws_iam_role" "service_codepipeline_role" {
  name = "${var.service_id}-codepipeline-service-role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "codepipeline.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF
}

# Codepipeline Policy. This looks like a ridiculous perms set but it's basically what
# codepipeline generates automatically if you create a pipeline through the GUI
resource "aws_iam_policy" "service_codepipeline_allow" {
  name   = "codepipeline-${var.service_id}-allow"
  policy = file("${path.module}/files/codepipeline_service_policy.json")
}

# Give the pipeline access to poll the S3 bucket
resource "aws_iam_role_policy_attachment" "service_codepipeline_add_policy_attachment" {
  role       = aws_iam_role.service_codepipeline_role.name
  policy_arn = aws_iam_policy.service_codepipeline_allow.arn
}
