#########################################################################
# Use this module to create subdomain hosted zones and delegate the zone
#########################################################################

terraform {
  required_version = ">= 0.12"
}

# Stub us-west-2 since it's a required parameter
provider "aws" {
  region = "us-west-2"
}

# The subdomain account
# Assumes that the proper roles are already setup
provider "aws" {
  region = "us-west-2"
  alias  = "subdomain"

  assume_role {
    role_arn = "arn:aws:iam::${var.subdomain_account_id}:role/${var.subdomain_account_role}"
  }
}

module "defaults" {
  source = "../modules/defaults"
}

locals {
  envs         = module.defaults.envs
  regions      = module.defaults.regions
  root_domains = module.defaults.root_domains

  primary_name      = "${var.service}.${var.domain}"
  subdomain_name    = join(".", compact([var.component, var.region, var.env, local.primary_name]))
  subdomain_comment = "Delegated to ${var.subdomain_account_id} (${var.subdomain_account_name})"
}

# Poor man's validation
# Ensure we only create zones that conforms to the env, region, and domain naming conventions
resource "null_resource" "validate_env" {
  count = contains(local.envs, var.env) == true ? 0 : "Allowed env is (dev, staging, prod) only"
}

resource "null_resource" "validate_region" {
  count = contains(local.regions, var.region) == true ? 0 : "Please provide a valid AWS region. E.g. (us-west-2)"
}

resource "null_resource" "validate_domain" {
  count = contains(local.root_domains, var.domain) == true ? 0 : "Please provide a valid domain. (live-video.a2z.com|live-video-ext.a2z.com)"
}

resource "null_resource" "validate_account" {
  count = data.aws_caller_identity.account.account_id == var.primary_account_id ? 0 : "Please check that you are using the AWS credentials for the primary account for this domain."
}

data "aws_route53_zone" "primary" {
  name = local.primary_name
}

data "aws_caller_identity" "account" {}

# Create subdomain zone in the delegated account
module "subdomain" {
  source = "../modules/zone"

  providers = {
    aws = aws.subdomain
  }

  name    = local.subdomain_name
  comment = "Delegated by account: ${data.aws_caller_identity.account.account_id}"
}

# Create required subdomain records
resource "aws_route53_record" "subdomain" {
  zone_id = data.aws_route53_zone.primary.zone_id
  name    = local.subdomain_name
  type    = "NS"
  ttl     = var.ttl

  records = module.subdomain.zone.name_servers

  depends_on = [null_resource.validate_account]
}

# Create a txt entry in route53 to as a way to "store" metadata when viewing DNS records in console.aws.amazon.com
resource "aws_route53_record" "comment" {
  zone_id = data.aws_route53_zone.primary.zone_id
  name    = "_comment.${local.subdomain_name}"
  type    = "TXT"
  ttl     = var.ttl

  records = [local.subdomain_comment]
}
