module "mapping" {
  source = "../mapping"
}


locals {
  transpose_pop_region = transpose(module.mapping.all_dcs)
  # Form raw list of maps where each element is {"pop": <pop-val>, "acc_id": <acc_id|milliner_arn>}
  # This nonsense is to circumvent tf errors if a mapping of prod/staging/dev is not defined
  pmap_raw = flatten([for region, acc_id in var.account_mapping["prod"] : [
    for pop in module.mapping.global_map[var.location] : {
      pop    = pop
      acc_id = var.account_mapping["prod"][join("", local.transpose_pop_region[pop])]
    }
  ]])
  smap_raw = flatten([for region, acc_id in var.account_mapping["staging"] : [
    for pop in module.mapping.global_map[var.location] : {
      pop    = pop
      acc_id = var.account_mapping["staging"][join("", local.transpose_pop_region[pop])]
    }
  ]])
  dmap_raw = flatten([for region, acc_id in var.account_mapping["dev"] : [
    for pop in module.mapping.global_map[var.location] : {
      pop    = pop
      acc_id = var.account_mapping["dev"][join("", local.transpose_pop_region[pop])]
    }
  ]])
  # key => PoP, value => AWS account ID or ARN of milliner IAM role
  prod_map = zipmap(
    flatten([for i in local.pmap_raw : i["pop"]]),
    flatten([for i in local.pmap_raw : i["acc_id"]])
  )
  staging_map = zipmap(
    flatten([for i in local.smap_raw : i["pop"]]),
    flatten([for i in local.smap_raw : i["acc_id"]])
  )
  dev_map = zipmap(
    flatten([for i in local.dmap_raw : i["pop"]]),
    flatten([for i in local.dmap_raw : i["acc_id"]])
  )
}

#=============================================================================
# Define the specific assumeRole policy for the roles

# Create the policies (based off of reading existing policies in sandstorm)
data "aws_iam_policy_document" "sandstorm_assume_role_policy_prod" {
  for_each = local.prod_map
  # Correct AssumeRole to milliner service account role
  statement {
    actions = ["sts:AssumeRole"]

    # Either deterministically form the milliner role or use one copy pasted from input variables
    principals {
      type        = "AWS"
      identifiers = length(regexall("[0-9]{12}", each.value)) > 0 ? ["arn:aws:iam::${each.value}:role/milliner-${var.machine_class}-prod-${each.key}"] : [each.value]
    }
  }
}
data "aws_iam_policy_document" "sandstorm_assume_role_policy_staging" {
  for_each = local.staging_map
  # Correct AssumeRole to milliner service account role
  statement {
    actions = ["sts:AssumeRole"]

    # Either deterministically form the milliner role or use one copy pasted from input variables
    principals {
      type        = "AWS"
      identifiers = length(regexall("[0-9]{12}", each.value)) > 0 ? ["arn:aws:iam::${each.value}:role/milliner-${var.machine_class}-staging-${each.key}"] : [each.value]
    }
  }
}
data "aws_iam_policy_document" "sandstorm_assume_role_policy_dev" {
  for_each = local.dev_map
  # Correct AssumeRole to milliner service account role
  statement {
    actions = ["sts:AssumeRole"]

    # Either deterministically form the milliner role or use one copy pasted from input variables
    principals {
      type        = "AWS"
      identifiers = length(regexall("[0-9]{12}", each.value)) > 0 ? ["arn:aws:iam::${each.value}:role/milliner-${var.machine_class}-dev-${each.key}"] : [each.value]
    }
  }
}


#=============================================================================
# Define the specific policies that give access to the correct dynamo paths
data "aws_iam_policy_document" "sandstorm_policy_prod" {
  for_each = local.prod_map
  # Dynamo Query for video bucket
  statement {
    sid       = "DynamoDBNamespaceIndex"
    actions   = ["dynamodb:Query"]
    resources = ["arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-production/index/namespace_name"]

    condition {
      test     = "StringEquals"
      variable = "dynamodb:LeadingKeys"
      values   = ["video"]
    }
  }
  # Dynamo permissions for machine class in env in DC specific secrets
  statement {
    sid = "DynamoDBSecretsTable"
    actions = [
      "dynamodb:Query",
      "dynamodb:DescribeTable",
      "dynamodb:GetItem"
    ]
    resources = ["arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-production"]

    condition {
      test     = "StringLike"
      variable = "dynamodb:LeadingKeys"
      values   = ["video/${var.machine_class}/prod/${each.key}/*"]
    }
  }
}
data "aws_iam_policy_document" "sandstorm_policy_staging" {
  for_each = local.staging_map
  # Dynamo Query for video bucket
  statement {
    sid       = "DynamoDBNamespaceIndex"
    actions   = ["dynamodb:Query"]
    resources = ["arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-production/index/namespace_name"]

    condition {
      test     = "StringEquals"
      variable = "dynamodb:LeadingKeys"
      values   = ["video"]
    }
  }
  # Dynamo permissions for machine class in env in DC specific secrets
  statement {
    sid = "DynamoDBSecretsTable"
    actions = [
      "dynamodb:Query",
      "dynamodb:DescribeTable",
      "dynamodb:GetItem"
    ]
    resources = ["arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-production"]

    condition {
      test     = "StringLike"
      variable = "dynamodb:LeadingKeys"
      values   = ["video/${var.machine_class}/staging/${each.key}/*"]
    }
  }
}
data "aws_iam_policy_document" "sandstorm_policy_dev" {
  for_each = local.dev_map
  # Dynamo Query for video bucket
  statement {
    sid       = "DynamoDBNamespaceIndex"
    actions   = ["dynamodb:Query"]
    resources = ["arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-production/index/namespace_name"]

    condition {
      test     = "StringEquals"
      variable = "dynamodb:LeadingKeys"
      values   = ["video"]
    }
  }
  # Dynamo permissions for machine class in env in DC specific secrets
  statement {
    sid = "DynamoDBSecretsTable"
    actions = [
      "dynamodb:Query",
      "dynamodb:DescribeTable",
      "dynamodb:GetItem"
    ]
    resources = ["arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-production"]

    condition {
      test     = "StringLike"
      variable = "dynamodb:LeadingKeys"
      values   = ["video/${var.machine_class}/dev/${each.key}/*"]
    }
  }
}


#=============================================================================
# Define the Roles
resource "aws_iam_role" "sandstorm_roles_prod" {
  for_each           = local.prod_map
  path               = "/sandstorm/production/templated/role/"
  name               = "sandstorm-milliner-${var.machine_class}-prod-${each.key}"
  assume_role_policy = data.aws_iam_policy_document.sandstorm_assume_role_policy_prod[each.key].json
}

resource "aws_iam_role" "sandstorm_roles_staging" {
  for_each           = local.staging_map
  path               = "/sandstorm/production/templated/role/"
  name               = "sandstorm-milliner-${var.machine_class}-staging-${each.key}"
  assume_role_policy = data.aws_iam_policy_document.sandstorm_assume_role_policy_staging[each.key].json
}

resource "aws_iam_role" "sandstorm_roles_dev" {
  for_each           = local.dev_map
  path               = "/sandstorm/production/templated/role/"
  name               = "sandstorm-milliner-${var.machine_class}-dev-${each.key}"
  assume_role_policy = data.aws_iam_policy_document.sandstorm_assume_role_policy_dev[each.key].json
}


#=============================================================================
# Define the policies
resource "aws_iam_policy" "sandstorm_policy_prod" {
  for_each = local.prod_map
  path     = "/sandstorm/production/templated/policy"
  name     = "sandstorm-milliner-${var.machine_class}-prod-${each.key}"
  policy   = data.aws_iam_policy_document.sandstorm_policy_prod[each.key].json
}
resource "aws_iam_policy" "sandstorm_policy_staging" {
  for_each = local.staging_map
  path     = "/sandstorm/production/templated/policy"
  name     = "sandstorm-milliner-${var.machine_class}-staging-${each.key}"
  policy   = data.aws_iam_policy_document.sandstorm_policy_staging[each.key].json
}
resource "aws_iam_policy" "sandstorm_policy_dev" {
  for_each = local.dev_map
  path     = "/sandstorm/production/templated/policy"
  name     = "sandstorm-milliner-${var.machine_class}-dev-${each.key}"
  policy   = data.aws_iam_policy_document.sandstorm_policy_dev[each.key].json
}


#=============================================================================
# Attach the policies (pre-defined for all roles in sandstorm)
resource "aws_iam_role_policy_attachment" "sandstorm_policy_attach_prod" {
  for_each   = local.prod_map
  role       = aws_iam_role.sandstorm_roles_prod[each.key].name
  policy_arn = aws_iam_policy.sandstorm_policy_prod[each.key].arn
}
resource "aws_iam_role_policy_attachment" "sandstorm_policy_attach_staging" {
  for_each   = local.staging_map
  role       = aws_iam_role.sandstorm_roles_staging[each.key].name
  policy_arn = aws_iam_policy.sandstorm_policy_staging[each.key].arn
}
resource "aws_iam_role_policy_attachment" "sandstorm_policy_attach_dev" {
  for_each   = local.dev_map
  role       = aws_iam_role.sandstorm_roles_dev[each.key].name
  policy_arn = aws_iam_policy.sandstorm_policy_dev[each.key].arn
}


#=============================================================================
# Attach the aux policies common to all sandstorm roles(pre-defined for all roles in sandstorm)
resource "aws_iam_role_policy_attachment" "sandstorm_default_policy_attach_prod" {
  for_each   = local.prod_map
  role       = aws_iam_role.sandstorm_roles_prod[each.key].name
  policy_arn = "arn:aws:iam::734326455073:policy/sandstorm/production/aux_policy/sandstorm-agent-production-aux"
}
resource "aws_iam_role_policy_attachment" "sandstorm_default_policy_attach_staging" {
  for_each   = local.staging_map
  role       = aws_iam_role.sandstorm_roles_staging[each.key].name
  policy_arn = "arn:aws:iam::734326455073:policy/sandstorm/production/aux_policy/sandstorm-agent-production-aux"
}
resource "aws_iam_role_policy_attachment" "sandstorm_default_policy_attach_dev" {
  for_each   = local.dev_map
  role       = aws_iam_role.sandstorm_roles_dev[each.key].name
  policy_arn = "arn:aws:iam::734326455073:policy/sandstorm/production/aux_policy/sandstorm-agent-production-aux"
}
