locals {
  environment_renderer_bucket_arn = "arn:aws:s3:::puppet-environment-renderer-bucket" # only one source bucket
}
data "archive_file" "unpack_zip" {
  type        = "zip"
  source_file = "${path.module}/unpack.py"
  output_path = "${path.module}/unpack.zip"
}

data "archive_file" "cleanup_zip" {
  type        = "zip"
  source_file = "${path.module}/cleanup.py"
  output_path = "${path.module}/cleanup.zip"
}

resource "aws_lambda_function" "puppet-environment-cleanup" {
  file_system_config {
    # EFS file system access point ARN
    arn = aws_efs_access_point.access_point_for_lambda.arn
    # Local mount path inside the lambda function. Must start with '/mnt/'.
    local_mount_path = "/mnt/efs"
  }

  vpc_config {
    security_group_ids = var.security_group_ids
    subnet_ids         = var.subnet_ids
  }
  filename         = data.archive_file.cleanup_zip.output_path
  function_name    = "puppet-environment-cleanup"
  role             = aws_iam_role.iam_for_puppet_environment_renderer_lambda.arn
  handler          = "cleanup.handler"
  source_code_hash = data.archive_file.cleanup_zip.output_base64sha256
  runtime          = "python3.8"
  timeout          = 600
}

resource "aws_lambda_function" "puppet-environment-renderer" {
  file_system_config {
    # EFS file system access point ARN
    arn = aws_efs_access_point.access_point_for_lambda.arn
    # Local mount path inside the lambda function. Must start with '/mnt/'.
    local_mount_path = "/mnt/efs"
  }

  vpc_config {
    security_group_ids = var.security_group_ids
    subnet_ids         = var.subnet_ids
  }
  filename         = data.archive_file.unpack_zip.output_path
  function_name    = "puppet-environment-renderer"
  role             = aws_iam_role.iam_for_puppet_environment_renderer_lambda.arn
  handler          = "unpack.handler"
  source_code_hash = data.archive_file.unpack_zip.output_base64sha256
  runtime          = "python3.8"
  timeout          = 600
}

resource "aws_iam_role" "iam_for_puppet_environment_renderer_lambda" {
  name = "iam_for_puppet_environment_renderer_lambda"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_policy" "default_iam_policy" {
  name        = "puppet-environment-renderer"
  path        = "/"
  description = "EFS related permissions"
  policy      = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "ec2:DescribeNetworkInterfaces",
        "ec2:CreateNetworkInterface",
        "ec2:DeleteNetworkInterface",
        "ec2:DescribeInstances",
        "ec2:AttachNetworkInterface",
        "sqs:SendMessage"
      ],
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "detault_iam_role_attachment" {
  role       = aws_iam_role.iam_for_puppet_environment_renderer_lambda.name
  policy_arn = aws_iam_policy.default_iam_policy.arn
}

resource "aws_iam_role_policy_attachment" "puppet-environment-renderer" {
  role       = aws_iam_role.iam_for_puppet_environment_renderer_lambda.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
}

resource "aws_iam_role_policy_attachment" "puppet-environment-renderer-s3" {
  role       = aws_iam_role.iam_for_puppet_environment_renderer_lambda.name
  policy_arn = "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess"
}

resource "aws_lambda_permission" "allow_bucket" {
  statement_id   = "AllowExecutionFromS3Bucket"
  action         = "lambda:InvokeFunction"
  function_name  = aws_lambda_function.puppet-environment-renderer.arn
  principal      = "s3.amazonaws.com"
  source_arn     = local.environment_renderer_bucket_arn
  source_account = "181769184623"
}
