##############################################
# Ownership, alerting, environment information
##############################################

variable "owner" {
  type        = string
  description = "The email address of the owner of this account. E.g. (video-coreservices@twitch.tv)"
}

variable "team" {
  type        = string
  description = "The name of the team owning this infrastructure. Name should all be lowercased, words separated by dash (-). (E.g. vidcs)"

  validation {
    condition     = can(regex("^([a-z])([a-z]|\\d|-)+$", var.team))
    error_message = "The name should all be lowercased, words separated by dash (-)."
  }
}

variable "env" {
  type        = string
  description = "The stack environment. One of dev, staging, or prod"
  default     = "prod"

  validation {
    condition     = can(regex("^(prod|staging|dev)$", var.env))
    error_message = "The variable env must be one of dev, staging, or prod."
  }
}

variable "pagerduty_url" {
  type        = string
  description = "The pagerduty cloudwatch integration URL used to alert teams of issues"
}

######################################
# VPC and subnets setup
######################################
variable "region" {
  type        = string
  description = "The AWS region to deploy this stack to"
  default     = "us-west-2"
}

variable "vpc_id" {
  type        = string
  description = "The VPC to deploy this stack to"
}

variable "subnets" {
  type        = list(string)
  description = "The subnets to deploy this stack to"
}

variable "security_groups" {
  type        = list(string)
  description = "The security groups to associate this stack to. Ensure that this security group permits traffic from sources that will be sending traffic to this cluster. Typically twitch subnets."
}

######################################
# Cluster components container images
######################################
variable "statsd_proxy_image_tag" {
  type        = string
  description = "The image hash/tag that will be used to run statsd-proxy"
  default     = "bootstrap"
}

variable "statsite_image_tag" {
  type        = string
  description = "The image hash/tag that will be used to run statsite"
  default     = "8dd7750dbecee9ca5fa29469d9e223cbbcf687c3"
}

variable "carbon_c_relay_image_tag" {
  type        = string
  description = "The image hash/tag that will be used to run carbon-c-relay"
  default     = "v3.7.2"
}

variable "go_carbon_image_tag" {
  type        = string
  description = "The image hash/tag that will be used to run go_carbon"
  default     = "v0.15.6"
}

variable "graphite_web_image_tag" {
  type        = string
  description = "The image hash/tag that will be used to run graphite-web"
  default     = "1ca8e602c15ba9ed7737976d01c9d1ba10eafa40"
}

######################################
# Cluster components sizing
######################################

variable "statsd_proxy_cluster_size" {
  type        = number
  description = "The desired cluster size for statsd-proxy"
  default     = 9
}

variable "statsd_cluster_size" {
  type        = number
  description = "The desired cluster size for statsd (statsite)"
  default     = 12
}

variable "carbon_c_relay_cluster_size" {
  type        = number
  description = "The desired cluster size for carbon-c-relay"
  default     = 8
}

variable "go_carbon_cluster_size" {
  type        = number
  description = <<EOT
  ATTENTION: this should be a fixed size, and not be changed after the cluster has been provisioned. Size cluster accordingly.
  The desired cluster size for each go-carbon cluster replica. I.e. The total number of instances deployed will double this number provided."
  EOT
}

variable "graphite_web_cluster_size" {
  type        = number
  description = "The desired cluster size for statsd-proxy"
  default     = 2
}

######################################
# Misc
######################################

variable "vpce_allowed_principals" {
  type        = list(any)
  description = "A list of trusted accounts for the graphite-web VPCe. Defaults to aws-infra prod and dev grafana accounts"
  default = [
    "arn:aws:iam::109561199089:root",
    "arn:aws:iam::963768028156:root",
  ]
}
