# Setup IAM to allow services to register w/ the ALB
resource "aws_iam_role" "ecs_service" {
  name               = "${var.team}-${var.environment}-ecs-service"
  assume_role_policy = data.aws_iam_policy_document.ecs_service.json
  description        = "Allows the service to register tasks with the ALB"
}

data "aws_iam_policy_document" "ecs_service" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_service" {
  name = "${var.team}-${var.environment}-ecs-service"
  role = aws_iam_role.ecs_service.name
}

resource "aws_iam_role_policy_attachment" "ecs_service_attach_policy" {
  role       = aws_iam_role.ecs_service.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceRole"
}

# Setup IAM to allow task autoscaling
resource "aws_iam_role" "ecs_autoscale" {
  name               = "${var.team}-${var.environment}-ecs-task-autoscale"
  description        = "Used by AWS's built in task auto scaler to modify running task count on cloudwatch alarms"
  assume_role_policy = data.aws_iam_policy_document.ecs_autoscale.json
}

data "aws_iam_policy_document" "ecs_autoscale" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["application-autoscaling.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_autoscale" {
  name = "${var.team}-${var.environment}-ecs-autoscale"
  role = aws_iam_role.ecs_autoscale.name
}

resource "aws_iam_role_policy_attachment" "ecs_autoscale_attach_policy" {
  role       = aws_iam_role.ecs_autoscale.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceAutoscaleRole"
}

# Setup IAM to allow hosts to update ECS deployed tasks

resource "aws_iam_role" "ecs_host" {
  description        = "Allows EC2 instance to be part of an ECS cluster"
  name               = "${var.team}-${var.environment}-ecs-host"
  assume_role_policy = data.aws_iam_policy_document.ecs_host.json
}

data "aws_iam_policy_document" "ecs_host" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_host" {
  name = "${var.team}-${var.environment}-ecs-host"
  role = aws_iam_role.ecs_host.name
}

# This lets the host run the ecs-agent and be part of an ECS cluster
resource "aws_iam_role_policy_attachment" "ecs_host_attach_policy" {
  role       = aws_iam_role.ecs_host.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceforEC2Role"
}

# This allows the host to run the SSM agent, which lets us roll out docker and ecs-agent updates
resource "aws_iam_role_policy_attachment" "ecs_host_ssm_policy" {
  role       = aws_iam_role.ecs_host.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM"
}

### ----- task level iam policies -----
resource "aws_iam_role" "xray_task" {
  name               = "${var.team}-${var.environment}-xray"
  description        = "Allows sending xray stats to the web AWS account"
  assume_role_policy = data.aws_iam_policy_document.xray_task.json
}

data "aws_iam_policy_document" "xray_task" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

resource "aws_iam_role_policy" "policy" {
  name = "${var.team}-${var.environment}-xray-assume-role"
  role = aws_iam_role.xray_task.id

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": "sts:AssumeRole",
      "Resource": ["${var.xray_role}"]
    }
  ]
}
EOF

}
