# Resources in this file are only created if we want to back the NLB with an ALB of our specification.

data "aws_lb" "input_alb" {
  arn = var.alb_arn
}

resource "aws_s3_bucket" "alb_ip_addr_bucket" {
  bucket_prefix = "${var.service_id}-${var.region}-nlb-alb-lambda"
  tags          = var.tags
}

resource "aws_lambda_function" "nlb_alb_lambda" {
  filename         = "${path.module}/lambda/populate_NLB_TG_with_ALB.zip"
  function_name    = "${var.service_id}-${var.region}-nlb-alb-lambda"
  source_code_hash = filebase64sha256("${path.module}/lambda/populate_NLB_TG_with_ALB.zip")
  role             = aws_iam_role.nlb_alb_lambda_role.arn
  handler          = "lambda.lambda_handler"
  runtime          = "python3.6"
  timeout          = 30
  tags             = var.tags
  environment {
    variables = {
      ALB_DNS_NAME                      = data.aws_lb.input_alb.dns_name
      ALB_LISTENER                      = var.alb_listener_port
      S3_BUCKET                         = aws_s3_bucket.alb_ip_addr_bucket.id
      NLB_TG_ARN                        = var.nlb_tg_arn
      MAX_LOOKUP_PER_INVOCATION         = 50
      INVOCATIONS_BEFORE_DEREGISTRATION = 3
      CW_METRIC_FLAG_IP_COUNT           = true
    }
  }
}


resource "aws_cloudwatch_event_rule" "every_one_minute" {
  name                = "${var.service_id}-${var.region}-every-minute"
  description         = "Fires every one minutes"
  schedule_expression = "rate(1 minute)"
  tags                = var.tags
}

resource "aws_cloudwatch_event_target" "check_foo_every_one_minute" {
  rule      = aws_cloudwatch_event_rule.every_one_minute.name
  target_id = "lambda"
  arn       = aws_lambda_function.nlb_alb_lambda.arn
}

resource "aws_lambda_permission" "nlb_alb_lambda_allow_cloudwatch" {
  statement_id  = "AllowExecutionFromCloudWatch"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.nlb_alb_lambda.function_name
  principal     = "events.amazonaws.com"
  source_arn    = aws_cloudwatch_event_rule.every_one_minute.arn
}

resource "aws_iam_role" "nlb_alb_lambda_role" {
  name               = "${var.service_id}-${var.region}-nlb-alb-lambda-role"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
  tags               = var.tags
}

resource "aws_iam_policy" "nlb_alb_lambda_policy" {
  name   = "${var.service_id}-${var.region}-nlb-alb-policy"
  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [{
      "Sid": "LambdaLogging",
      "Effect": "Allow",
      "Action": [
        "logs:CreateLogGroup",
        "logs:CreateLogStream",
        "logs:PutLogEvents"
      ],
      "Resource": [
        "arn:aws:logs:*:*:*"
      ]
    },
    {
      "Sid": "S3",
      "Action": [
        "s3:Get*",
        "s3:PutObject",
        "s3:CreateBucket",
        "s3:ListBucket",
        "s3:ListAllMyBuckets"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_s3_bucket.alb_ip_addr_bucket.arn}",
        "${aws_s3_bucket.alb_ip_addr_bucket.arn}/*"
      ]
    },
    {
      "Sid": "ELB",
      "Action": [
        "elasticloadbalancing:RegisterTargets",
        "elasticloadbalancing:DeregisterTargets"
      ],
      "Effect": "Allow",
      "Resource": [
        "${var.nlb_arn}",
        "${var.nlb_tg_arn}"
      ]
    },
    {
      "Sid": "DescribeELB",
      "Action": [
        "elasticloadbalancing:Describe*"
      ],
      "Effect": "Allow",
      "Resource": "*"
    },
    {
      "Sid": "CW",
      "Action": [
        "cloudwatch:putMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "nlb_alb_lambda_rp_attach" {
  role       = aws_iam_role.nlb_alb_lambda_role.name
  policy_arn = aws_iam_policy.nlb_alb_lambda_policy.arn
}
