# tp

tp (Twitch proxy) is a script to make network requests to the Twitch network when connected to the Amazon network (WPA2). The proxy used is [Video Bastion](https://wiki.twitch.com/display/VID/Video+Bastion).

You should use this if you want to be connected to the Amazon network all the time and still access resources on the Twitch network, such as Go dependencies on code.justin.tv, or consul.

## Installation

1. Add [tp](tp) to your PATH

    ```bash
    export PATH="/path/to/tp:$PATH"
    ```
2. Install required tools:
   * `brew install coreutils` for CLI tools
   * `npm install -g http-proxy-to-socks` for local http proxy (NOT SOCKS proxy) tools

3. Follow the one-time setup on [Video Bastion](https://wiki.twitch.com/display/VID/Video+Bastion). Step 4  can be skipped (`Add the following directives to your $HOME/.ssh/config`). tp will add and remove ssh config lines on enable and disable respectively.

## Usage

### Daily

* `tp enable` Enable proxy connection. Authenticates if needed. Restarts if already enabled
* `tp disable` Disable proxy connection. You may want to do this if you are enabling teleport-bastion
* `tp status` Check proxy connection

After enabling tp:
* `tp proxy` can be used to run commands against the Twitch network that support socks proxying
* `tp ssh` can be used to run commands on the bastion that's connected to the Twitch network
* `go` tool can be used to fetch code.justin.tv dependencies. Note: `tp proxy go` is unnecessary on the Amazon network because https://code.justin.tv is available on the network without a proxy.
* Repositories on GHE can be cloned. `git@git.xarth.tv:...`

**If you have teleport-bastion enabled, disable it, because it will interfere with connecting to the Video Bastion.**

### Switching between Video Bastion and teleport bastion

Switching between using Video Bastion and teleport bastion can be done while connected to the Amazon network

Switch to teleport bastion
1. `tp disable`
1. `teleport-bastion enable && teleport-bastion login`

Switch to Video Bastion
1. `teleport-bastion disable`
1. `tp enable`

### Accessing Twitch-only network websites on the Amazon network

Google Chrome solution: See this [wiki comment](https://wiki.twitch.com/display/VID/Video+Bastion?focusedCommentId=181471122#comment-181471122) to use the [Proxy SwitchOmega](https://chrome.google.com/webstore/detail/proxy-switchyomega/padekgcemlokbadohgkifijomclgjgif) extension. For example, accessing
https://aws-vpc-api.internal.justin.tv/ is possible on the Amazon network.

### Golang Integration

A golang application can make use of Video Bastion to make HTTP calls to services on the Twitch network. If the Golang HTTP client transport uses [proxy from environment](https://github.com/golang/go/blob/729930869064b1c6989b70ef8517a101d42c7f9b/src/net/http/transport.go#L43), then the HTTP client will consider the env variable `HTTP_PROXY`. `HTTP_PROXY` can be configured to use a SSH SOCKS5 proxy, which means the Golang HTTP client will make DNS and TCP connections through the SSH connection.

Proof of concept
```go
package main

import (
	"context"
	"fmt"
	"log"
	"net/http"
	"time"
)

func main() {
	ctx, cancel := context.WithTimeout(context.Background(), 1*time.Second)
	defer cancel()
	req, err := http.NewRequestWithContext(ctx, "GET", "http://dev.users-service.twitch.a2z.com/users/111", nil)
	if err != nil {
		log.Fatalf("error making request: %v", err)
	}
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		log.Fatalf("error getting response: %v", err)
	}

	fmt.Println(resp.StatusCode)
}
```

`go run main.go` on the Amazon network will print
```
2020/02/19 13:39:53 error getting response: Get http://dev.users-service.twitch.a2z.com/users/111: context deadline exceeded
exit status 1
```

`tp proxy go run main.go` on the Amazon network will print
```
200
```


### Commands

#### Proxy

##### SOCKS Proxy

Execute a shell command using the socks proxy. The program should support socks proxying via environment variables (`ALL_PROXY`, `HTTP_PROXY`).

```bash
tp proxy [command]
```

Examples
```bash
# curl users-service
tp proxy curl https://dev.users-service.twitch.a2z.com/users/1 # shortcut with `tp curl`
# Run lsconsul (Go program)
tp proxy lsconsul -x consul
```

##### HTTP Proxy

Execute a shell command using the HTTP proxy. The program should support http proxying via environment variables (`HTTP_PROXY`, `HTTPS_PROXY`, `http_proxy`, `https_proxy`). This is for an application (such as Docker) that does not support SOCKS proxy.

```bash
tp http_proxy [command]
```

Examples
```bash
# curl users-service
tp http_proxy curl https://dev.users-service.twitch.a2z.com/users/1
# Run lsconsul (Go program)
tp http_proxy lsconsul -x consul
```

#### SSH

Execute a shell command on the bastion host.

```bash
tp ssh [command]
```

Examples
```bash
# Lookup DNS
tp ssh nslookup consul.internal.justin.tv
```

#### Enable

Enable the proxy connection
```bash
tp enable
```

#### Disable

Disable the proxy tunnel
```bash
tp disable
```

#### Status

Check if the proxy tunnel is working, and if authenticated to midway.
```bash
tp status
```

#### curl

Shortcut for `tp proxy curl`
```bash
tp curl [args]
```

## Credits

This tool was inspired by [tgoproxy](https://git.xarth.tv/twitch/tgoproxy)
