#!/bin/bash

set -ue

script_dir="$(dirname $(realpath $0))"
output_dir="$script_dir/output"
wait_seconds=60
ssh_config=$output_dir/ssh_config

run_ssh() {
    ssh -F $ssh_config qemu "$@"
}

get_free_port() {
    local port
    while true; do
        port=$((1024 + RANDOM % 8192))
        netstat -lnt|grep -q ":${port}\b" && continue
        echo $port
        return
    done
}

kill_pids() {
    local pids=$(cat $output_dir/*.pid 2>/dev/null)
    [ -n "$pids" ] && kill ${1:-} $pids >/dev/null 2>&1
}

start() {
    kill_pids -0 && echo "Already running" && exit
    http_port=$(get_free_port)
    $script_dir/prepare-boot.sh "http://10.0.2.2:$http_port"
    cd $output_dir
    python3 -m http.server $http_port >http.log 2>&1 &
    echo $! >http.pid
    domainname="$(hostname -d)"
    ssh_port=$(get_free_port)
    printf "%s\n" \
        "Host qemu" \
        "  HostName 127.0.0.1" \
        "  User toor" \
        "  Port $ssh_port" \
        "  UserKnownHostsFile /dev/null" \
        "  StrictHostKeyChecking no" \
        "  PasswordAuthentication no" \
        "  LogLevel FATAL" > $ssh_config
    qemu-system-x86_64 \
        -boot n \
        -device virtio-net-pci,netdev=n1 \
        -netdev user,id=n1,tftp=${output_dir},bootfile=/boot.ipxe,hostfwd=tcp::${ssh_port}-:22,domainname=${domainname} \
        -nographic \
        -m 4096 \
        -enable-kvm \
        -cpu max >qemu.log 2>&1 &

    echo $! >qemu.pid
    deadline=$((SECONDS + wait_seconds))
    while ((SECONDS < deadline)); do
        run_ssh true >/dev/null 2>&1 && echo "VM is ready" && exit
        sleep 5
    done
    echo "VM failed to start"
}

case ${1:-} in
    start) start ;;
    stop) kill_pids || true ;;
    ssh) shift && run_ssh "$@" ;;
    *) echo "Usage: $0 start|stop|ssh" ;;
esac
