#!/usr/bin/env bash
set -e

OWNER="video-coreservices"
REPO="video-cloudwatch-agent"

usage() {
	echo "Usage: $0 -b <branch> -e <environment> -c <commit> -t <access token>" 1>&2
	exit 1
}

while getopts ":b:c:e:t:" o; do
	case "${o}" in
		b)
			BRANCH=${OPTARG}
			;;
		c)
			GIT_COMMIT=${OPTARG}
			;;
		e)
			AUTO_DEPLOY_ENV=${OPTARG}
			;;
		t)
			GITHUB_ACCESS_TOKEN=${OPTARG}
			;;
		*)	usage
			;;
	esac
done
shift $((OPTIND-1))

if [ -z "${AUTO_DEPLOY_ENV}" ] || [ -z "${BRANCH}" ] || [ -z "${GIT_COMMIT}" ] || [ -z "${GITHUB_ACCESS_TOKEN}" ]; then
	usage
fi

# Get previous deployed commit for `master`
LAST_DEPLOYMENT_DATA=$(curl "https://clean-deploy.internal.justin.tv/v1/artifact/compare?env=${AUTO_DEPLOY_ENV}&name=${REPO}&owner=${OWNER}&ref=master");
REGEXP='"previous_sha":"([^"]+)"'
if [[ $LAST_DEPLOYMENT_DATA =~ $REGEXP ]]
then
  CODE_REVIEW_URL="https://git.xarth.tv/${OWNER}/${REPO}/compare/${BASH_REMATCH[1]}...${GIT_COMMIT}";
else
  echo "Can not find previously deployed SHA for ${AUTO_DEPLOY_ENV}. Code review URL will only include the deploying commit.";
  CODE_REVIEW_URL="https://git.xarth.tv/${OWNER}/${REPO}/commit/${GIT_COMMIT}";
fi
COMMIT_TITLE=`git show ${GIT_COMMIT} --format='%s' | head -1`

# Trigger a deployment via posting to the Skadi deployments endpoint. This is the same mechanism
# as is used by the deploy tool.
curl "https://clean-deploy.internal.justin.tv/v1/deployments" \
  -H "GithubAccessToken: ${GITHUB_ACCESS_TOKEN}" \
  --data-urlencode "description=${COMMIT_TITLE}" \
  --data-urlencode "owner=${OWNER}" \
  --data-urlencode "repo=${REPO}" \
  --data-urlencode "ref=${GIT_COMMIT}" \
  --data-urlencode "branch=${BRANCH}" \
  --data-urlencode "environment=${AUTO_DEPLOY_ENV}" \
  --data-urlencode "codeReviewURL=${CODE_REVIEW_URL}" \
  --data-urlencode "severity=auto-deploy"
