package main

/* glitch.go: contains bot-specific methods. */

import (
	// standard
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"strings"
	"syscall"
	"time"

	// external
	"github.com/gorilla/mux"
	"github.com/pkg/errors"

	// local

	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/handlers/exports"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/handlers/incidents"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/handlers/oncall"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/handlers/tickets"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/handlers/vidcs"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/pagerduty"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
)

// GetGlitch creates a fresh instance of a Glitch bot.
// Only happens once, does not re-occur on reload.
func GetGlitch(config *Config) *Glitch {
	glitch := &Glitch{
		Config: config,
		// Make a Gorilla router for handling HTTP requests.
		Router: mux.NewRouter(),
		// Create signal channels to catch a HUP or TERM.
		SigKILLChan: make(chan os.Signal, 1),
		SigHUPChan:  make(chan os.Signal, 1),
	}
	// initialize the config, and everything else.
	glitch.initialize()
	// Pass all routes to the gorilla Router.
	http.Handle("/", glitch.Router)
	// Make expvar route work with Gorilla.
	glitch.Router.Handle("/debug/vars", http.DefaultServeMux)
	// Add signal notifiers.
	signal.Notify(glitch.SigKILLChan, os.Interrupt, syscall.SIGTERM, syscall.SIGQUIT)
	signal.Notify(glitch.SigHUPChan, syscall.SIGHUP)
	return glitch
}

// RunForever watching for reload or terminate signal.
func (g *Glitch) RunForever() {
	for {
		select {
		case sigc := <-g.SigHUPChan:
			log.Print("Caught Signal (", sigc.String(), "), reloading!")
			if err := g.Stop(); err != nil {
				exit(g.Export.pidFile.String(), "Error Stopping Handler: "+err.Error(), 255)
			}
			// Reload the config.
			exp.Debug("Re-loading Configuration File.")
			g.Config = loadConfig(g.Config.configFile, g.Config.pidFile)
			// Start everything back up.
			exp.Debug("Re-Initializing.")
			g.initialize()
			g.Export.reloads.Add(1)
		case sigc := <-g.SigKILLChan:
			exit(g.Export.pidFile.Value(), "Caught Signal ("+sigc.String()+"), exiting!", 0)
		}
	}
}

// initialize is called by GetGlitch() and again after a reload.
func (g *Glitch) initialize() {
	if g.Config.Slack == nil {
		exit(g.Config.pidFile, "Error Loading Slack Configuration, make sure it's in the config file.", 5)
	} else { // else, so tests don't panic.
		// Pass the server hostname and glitch version into Slack so they can be used in client responses.
		g.Config.Slack.Hostname = Fqdn()
		g.Config.Slack.BotName = BotName
		if parts := strings.Split(g.Config.BindAddr, ":"); len(parts) > 1 {
			// Split the port out of the bind address and append it to the hostname to make a URL.
			g.Config.Slack.Hostname += ":" + parts[1]
		}
	}
	log.Println("Loaded Config File:", g.Config.configFile)

	// Create Slack Client, This kicks off a connection and Slack Watcher routine.
	g.SlackClient = getSlackClient(g.Config.Slack)
	// Initialize "root" expvar map. Must happen before initHandlers().
	g.Export.Map = exp.GetPublishedMap(BotName)
	// Fire up each handler. Must happen before exportVars().
	g.initHandlers()
	// Export Data Variables (gxpvar), requires handlers for slack.Dial().
	g.exportVars(g.Config.configFile, g.Config.pidFile, g.Config.BindAddr)
	// Start the HTTP Listener.
	g.createHTTPListener()
	log.Print(BotName, ": Startup Complete!")
}

// Stop everything! Used by HUP signal and in tests.
func (g *Glitch) Stop() error {
	// Stop Slack Connection.
	if g.SlackClient == nil {
		exp.Debug("Slack client not active, nothing to stop.")
	} else {
		exp.Debug("Stopping Slack Client")
		g.SlackClient.Stop()
	}

	// Stop HTTP Server.
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*2)
	defer cancel()
	// Doesn't block if no connections, otherwise wait 2 seconds.
	if g.HTTPServer == nil {
		exp.Debug("HTTP Server not active, nothing to stop.")
	} else if err := g.HTTPServer.Shutdown(ctx); err != nil {
		log.Println("Error Close()ing net/HTTP listener:", err)
	}

	// Stop Handlers.
	for _, handler := range g.Handlers {
		name := handler.Name()
		exp.Debug("Stopping handler: " + name)
		if err := handler.Stop(); err != nil {
			// Returning an error causes os.Exit()
			return errors.Wrap(err, name)
		}
	}

	return nil
}

// Export base data set. This also Dial()'s slack.
func (g *Glitch) exportVars(configFile, pidFile, bindAddr string) {
	g.Export.Set("slack", g.SlackClient.Dial(g.Handlers))
	g.Export.Set("glitch_version", &g.Export.version)
	g.Export.Set("http_visits", &g.Export.httpVisits)
	g.Export.Set("bind_addr", &g.Export.bindAddr)
	g.Export.Set("pid_file", &g.Export.pidFile)
	g.Export.Set("config_file", &g.Export.configFile)
	g.Export.Set("reloads", &g.Export.reloads)
	g.Export.Set("debug_log", exp.GetMap(exp.DebugMap))
	g.Export.bindAddr.Set(bindAddr)
	g.Export.version.Set(Version)
	g.Export.configFile.Set(configFile)
	g.Export.pidFile.Set(pidFile)
}

// Setup slack handlers. New handlers need to setup here.
func (g *Glitch) initHandlers() {
	// Reset to avoid duplicate handlers on reload.
	g.Handlers = []slack.Handler{
		// Append our default handlers.
		exports.Init(&exports.Config{BotName: BotName}, g.SlackClient),
	}
	// Only Init the following handlers if they have config data.

	// May add jira functionality to Vidcs module at later date, at that time this will
	// need to be passed jiraClient
	if g.Config.Vidcs != nil {
		//g.Handlers = append(g.Handlers, vidcs.Init(g.Config.Vidcs, g.SlackClient, jiraClient))
		g.Handlers = append(g.Handlers, vidcs.Init(g.Config.Vidcs, g.SlackClient))
	}
	if g.Config.OnCall != nil {
		pdClient := pagerduty.GetClient(g.Config.OnCall.APIToken)
		g.Handlers = append(g.Handlers, oncall.Init(g.Config.OnCall, g.SlackClient, pdClient))
	}
	if jiraClient, err := getJiraClient(g.Config.Jira); err == nil {
		// Only start the jira-dependent handlers if jira-client was successful.
		if g.Config.Incidents != nil {
			g.Handlers = append(g.Handlers, incidents.Init(g.Config.Incidents, g.SlackClient, jiraClient))
		}
		if g.Config.Tickets != nil {
			g.Handlers = append(g.Handlers, tickets.Init(g.Config.Tickets, g.SlackClient, jiraClient))
		}
		//if g.Config.Vidcs != nil {
		//	g.Handlers = append(g.Handlers, vidcs.Init(g.Config.Vidcs, g.SlackClient, jiraClient))
		//}
	} else {
		// Log and continue running anyway.
		log.Println("Error From jira.GetClient:", err)
	}
	// Export each handler's data, and Initialize the map if it's nil.
	for _, handler := range g.Handlers {
		if g.Export.Set(handler.Name(), handler.Export()); handler.Export() == nil {
			handler.Export().Init()
		}
	}
}
