package main

import (
	// standard
	"io/ioutil"
	"os"
	"testing"

	// external
	"github.com/stretchr/testify/assert"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/jira"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/jira/jirafakes"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack"
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/slack/slackfakes"
)

func TestGetGlitch(t *testing.T) {
	a := assert.New(t)
	// Create a fake config file.
	configFile := "/tmp/go_test_main_glitch_config_file.conf"
	a.Nil(os.RemoveAll(configFile), "error removing test conf file")
	a.Nil(ioutil.WriteFile(configFile, []byte(configData), 0644), "error writing test conf file")
	// Create fake jira and slack clients.
	getJiraClient = func(j *jira.Config) (jira.Client, error) {
		return new(jirafakes.FakeClient), nil
	}
	getSlackClient = func(s *slack.Config) slack.Client {
		return new(slackfakes.FakeClient)
	}
	// Run setup w/ minimal config data.
	c := &Config{Slack: &slack.Config{}}
	c.configFile = "/tmp/go_test_main_glitch_config_file.conf"
	glitch := GetGlitch(c)
	// Check data:
	// - Did jira and slack get called the right number of times?
	// - Do we have the right amount of handlers? (0)
	// - Make sure the channels returned are not nil.
	// - Test a reload signal.
	// Stop the handlers.
	a.Nil(glitch.Stop())
	// handlers.Stop() creates expvar debug data we can inspect/test
	// Signal an exit.
	glitch.SigKILLChan <- os.Interrupt
}

func TestStart(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	configFile := "/tmp/go_test_glitch_config_file.conf"

	a.Nil(os.RemoveAll(configFile), "error removing test conf file")
	a.Nil(ioutil.WriteFile(configFile, []byte(configData), 0644), "error writing test conf file")

	// Try to initialize with a plain-jane config.
	g := &Glitch{Config: &Config{configFile: configFile}}
	g.initialize()
	a.EqualValues(1, len(g.Handlers), "the config contained no handlers; only the defaults must be returned")

	// Now make initialize pull our fake mocked-data clients.
	getJiraClient = func(j *jira.Config) (jira.Client, error) {
		return new(jirafakes.FakeClient), nil
	}
	getSlackClient = func(s *slack.Config) slack.Client {
		return new(slackfakes.FakeClient)
	}
	// With Mock data, we get past the jira errors.
	g.initialize()
	a.EqualValues(1, len(g.Handlers), "the config contained no handlers; only the defaults must be returned")
	// Now add handler configs, mock different jira & slack data and check all the calls. (more testing...)
}
