package main

/* main.go: contains cli, config file, pid file and exit functions. */

import (
	// standard
	"fmt"
	"io/ioutil"
	"log"
	"net"
	"os"
	"strconv"
	"syscall"

	// external
	"github.com/naoina/toml"
	flg "github.com/ogier/pflag"

	// local
	"code.justin.tv/video-coreservices/video-coreservices-slack-bot/pkg/exp"
)

func main() {
	// Do not exit. Parse CLI flags, Get a Glitch and run forever!
	GetGlitch(setup(parseCliFlags())).RunForever()
}

// setup bridges parsing cli flags and starting the bot. Runs one time; does not reload.
func setup(pidFile, configFile string) *Config {
	log.Printf("%v (glitch v%v) Starting Up, PID: %v", BotName, Version, os.Getpid())
	writePidFile(pidFile)                  // If this function fails, it exits.
	return loadConfig(configFile, pidFile) // If this function fails, it exits.
}

// Parses the cli flags. May be overridden by tests.
// Sets botname and debug, returns pid file and configuration file paths.
func parseCliFlags() (string, string) {
	configFile := flg.StringP("config", "c", defaultConfigPath, "Config File Path")
	showVer := flg.BoolP("version", "v", false, "Display version and exit")
	pidFile := flg.StringP("pidfile", "p", defaultPidFilePath, "PID File Path")
	flg.BoolVarP(&exp.DebugLog, "debug", "D", false, "Enable Verbose Output")
	flg.StringVarP(&BotName, "botname", "b", BotName, "Bot Name. Not used for much yet")
	switch flg.Parse(); {
	case *showVer:
		fmt.Printf("glitch v%s\n", Version)
		os.Exit(0)
	case exp.DebugLog:
		log.SetFlags(log.Lshortfile | log.Lmicroseconds | log.Ldate)
		exp.Debug("Verbose Output Enabled")
	default:
		log.SetFlags(0)
	}
	return *pidFile, *configFile
}

// Check if the pid file exists and has a running pid then return it. Return 0 otherwise.
func getPidFromFile(pidFile string) (pid int) {
	// Read in the pid file as a slice of bytes.
	if piddata, err := ioutil.ReadFile(pidFile); err != nil {
		// Unreadable file, so try to write a new one.
		return
	} else if pid, err = strconv.Atoi(string(piddata)); err != nil {
		// File contents are not an integer, so try to write a new file.
		return 0
	} else if process, err := os.FindProcess(pid); err != nil {
		// On Unix systems, FindProcess always succeeds and returns a Process
		// for the given pid, regardless of whether the process exists.
		return 0
	} else if err := process.Signal(syscall.Signal(0)); err != nil {
		// Send the process a signal zero kill.
		// If we get here, then the pidfile didn't exist,
		// or the pid in it doesn't belong to the user running this app.
		return 0
	}
	return
}

// Write a pid file, but first make sure it doesn't exist with a running pid.
func writePidFile(pidFile string) (string, int) {
	if pidFile == "" {
		return "", 0
	} else if pid := getPidFromFile(pidFile); pid != 0 {
		return exit("", "Exiting! "+BotName+" Already Running with PID "+strconv.Itoa(pid), 3)
	} else if err := ioutil.WriteFile(pidFile, []byte(strconv.Itoa(os.Getpid())), 0644); err != nil {
		return exit("", "Error Writing Pid File "+pidFile+" - exiting: "+err.Error(), 2)
	}
	return "", 0
}

// loadConfig Unmarshals the config file.
func loadConfig(configFile, pidFile string) *Config {
	config := &Config{}
	if data, err := ioutil.ReadFile(configFile); err != nil {
		exit(pidFile, "Error Reading Config File: "+err.Error(), 1)
	} else if err = toml.Unmarshal(data, config); err != nil {
		exit(pidFile, "Error Parsing Config File: "+err.Error(), 4)
	}
	config.pidFile = pidFile
	config.configFile = configFile
	return config
}

// exit after cleanup.
func exit(pidFile, msg string, code int) (string, int) {
	if code != 0 {
		log.SetOutput(os.Stderr)
	}
	log.Print(msg)
	if pidFile != "" {
		if err := os.Remove(pidFile); err != nil {
			log.Printf("Error Removing PID File %v: %v", pidFile, err)
		}
	}
	if !testRun { // Do not exit when testing.
		os.Exit(code)
	}
	return msg, code
}

// Fqdn Returns the FQDN of this host.
func Fqdn() string {
	// non-fqdn hostname.
	hostname, err := os.Hostname()
	if err != nil {
		return "localhost"
	}

	// All the IPs our non-fqdn may resolve to.
	ips, err := net.LookupIP(hostname)
	if err != nil {
		return hostname
	}
	// Just use the first IP and make sure it's really an IP.
	ip := ips[0].To4()
	if ip == nil {
		return hostname
	}
	// Get all the hostnames our IP resolves to.
	hosts, err := net.LookupAddr(ip.String())
	if err != nil || len(hosts) < 1 {
		return hostname
	}
	// Just use the first hostname found, trim the dot off the end.
	return hosts[0][0 : len(hosts[0])-1]
}
