package main

import (
	// standard
	"io/ioutil"
	"os"
	"strconv"
	"testing"

	// external
	"github.com/stretchr/testify/assert"
)

var configData = `[slack]
	api_token = ""
	debug_channel = ""
	log_dir = "/var/log/glitch"

	# JIRA Connection configuration.
	[jira]
	base_url = "https://jira.twitch.com"
	access_token = ""
	access_token_secret = ""
	consumer_key = "glitch-bot"
	private_key_location = "/tmp/go_test_glitch_jira_private_key.pem"`

func init() {
	testRun = true
}

func TestGetPidFromFile(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	testPidFile := "/tmp/go_test_glitch_pid_file.pid"
	a.Nil(os.RemoveAll(testPidFile), "error removing test pid file")

	// Test missing PID file.
	pid := getPidFromFile(testPidFile)
	a.EqualValues(0, pid, "missing file must return 0 for a pid")

	// Test invalid PID.
	a.Nil(ioutil.WriteFile(testPidFile, []byte("foo"), 0644), "error writing test pid file")
	pid = getPidFromFile(testPidFile)
	a.EqualValues(0, pid, "invalid pid must return 0")

	// Test fake PID.
	a.Nil(ioutil.WriteFile(testPidFile, []byte("89167823"), 0644), "error writing test pid file")
	pid = getPidFromFile(testPidFile)
	a.EqualValues(0, pid, "not running pid must return 0")

	// Test real, running PID.
	a.Nil(ioutil.WriteFile(testPidFile, []byte(strconv.Itoa(os.Getpid())), 0644), "error writing test pid file")
	pid = getPidFromFile(testPidFile)
	a.EqualValues(os.Getpid(), pid, "test of running pid must return the pid")

	a.Nil(os.RemoveAll(testPidFile), "error removing test pid file")
}

func TestSetup(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	config := setup("", "string2")
	a.EqualValues("", config.pidFile)
	a.EqualValues("string2", config.configFile)
}

func TestWritePidFile(t *testing.T) {
	t.Parallel()
	a := assert.New(t)
	testPidFile := "/tmp/go_test_glitch_pid_file2.pid"
	a.Nil(os.RemoveAll(testPidFile), "error removing test pid file")

	// Test missing PID.
	msg, code := writePidFile("")
	a.EqualValues(0, code, "code must be 0 when no pid file is provided")
	a.EqualValues("", msg, "msg must be empty when no pid file is provided")

	// Test valid pid file creation.
	msg, code = writePidFile(testPidFile)
	a.EqualValues(0, code, "code must be 0 when pid file is created")
	a.EqualValues("", msg, "msg must be empty when pid file is created")
	// Check the contents of the written file.
	filePid, err := ioutil.ReadFile(testPidFile)
	a.Nil(err, "error reading test pid file")
	a.EqualValues(strconv.Itoa(os.Getpid()), string(filePid), "pid written to pid file is invalid")

	// Test already running pid.
	msg, code = writePidFile(testPidFile)
	a.EqualValues(3, code, "code must be 3 when pid file is already running")
	a.Contains(msg, BotName+" Already Running", "the error message must describe the problem")

	// Test invalid pid file path.
	msg, code = writePidFile("/tmp/this/must/not/exist/nor/work/in/any/way/XOXOXOXO")
	a.EqualValues(2, code, "code must be 2 when pid file is unwritable")
	a.Contains(msg, "Error Writing Pid File", "the error message must describe the problem")

	a.Nil(os.RemoveAll(testPidFile), "error removing test pid file")
}

func TestLoadConfig(t *testing.T) {
	t.Parallel()
	a := assert.New(t)

	// Test missing config file.
	config := loadConfig("/tmp/this/must/not/exist/bleedPurple", "")
	emptyConfig := &Config{pidFile: "", configFile: "/tmp/this/must/not/exist/bleedPurple"}
	a.EqualValues(emptyConfig, config, "passing in a missing config file must return an empty config")

	// Test invalid formatted config file, and deleting missing pid file.
	config = loadConfig("/etc/passwd", "/tmp/<--that/is/so/lulz")
	emptyConfig = &Config{pidFile: "/tmp/<--that/is/so/lulz", configFile: "/etc/passwd"}
	a.EqualValues(emptyConfig, config, "passing in an invalid config file must return an empty config")

	// Test valid config unmarshal.
	tomlData := "[slack]\napi_token = \"\"\ndebug_channel = \"\"\nlog_dir = \"find_me!\"\n"
	testTomlFile := "/tmp/go_test_glitch_toml_file.conf"
	a.Nil(ioutil.WriteFile(testTomlFile, []byte(tomlData), 0644), "error writing test toml file")
	config = loadConfig(testTomlFile, "")
	a.EqualValues("find_me!", config.Slack.LogDir, "values in config file must be present in reply")
	a.Nil(os.RemoveAll(testTomlFile), "error removing test config file")
}
